/**
 * P2P Protocol Handlers
 *
 * Handles Bob API protocol over P2P streams
 */

// Bob API Protocol ID
const BOB_API_PROTOCOL = '/bob-api/1.0.0';

// Maximum message size (10MB)
const MAX_MESSAGE_SIZE = 10 * 1024 * 1024;

/**
 * Read a message from a stream
 */
async function readMessage(stream) {
    const chunks = [];
    let totalSize = 0;

    try {
        for await (const chunk of stream.source) {
            totalSize += chunk.length;

            if (totalSize > MAX_MESSAGE_SIZE) {
                throw new Error(`Message exceeds maximum size of ${MAX_MESSAGE_SIZE} bytes`);
            }

            chunks.push(chunk);
        }

        const buffer = Buffer.concat(chunks);
        const message = buffer.toString('utf8');
        return JSON.parse(message);
    } catch (error) {
        if (error.message.includes('Message exceeds')) {
            throw error;
        }
        throw new Error(`Failed to read message: ${error.message}`);
    }
}

/**
 * Write a message to a stream
 */
async function writeMessage(stream, message) {
    try {
        const json = JSON.stringify(message);
        const buffer = Buffer.from(json, 'utf8');

        if (buffer.length > MAX_MESSAGE_SIZE) {
            throw new Error(`Message exceeds maximum size of ${MAX_MESSAGE_SIZE} bytes`);
        }

        // Write buffer to stream sink
        const writer = stream.sink((async function* () {
            yield buffer;
        })());

        await writer;
    } catch (error) {
        throw new Error(`Failed to write message: ${error.message}`);
    }
}

/**
 * Create API request handler for provider
 * This handles incoming API requests from consumers
 */
function createApiRequestHandler(queueManager) {
    return async (stream) => {
        try {
            // Read the request
            const request = await readMessage(stream);

            console.log(`Received API request: ${request.apiId}`, {
                jobId: request.jobId,
                consumer: request.consumer
            });

            // Validate request
            if (!request.apiId || !request.jobId || !request.consumer) {
                await writeMessage(stream, {
                    success: false,
                    error: 'Invalid request: missing required fields'
                });
                return;
            }

            // Find the API
            const api = queueManager.getApi(request.apiId);
            if (!api) {
                await writeMessage(stream, {
                    success: false,
                    error: `API not found: ${request.apiId}`
                });
                return;
            }

            // Create job entry
            const job = {
                id: request.jobId,
                apiId: request.apiId,
                consumer: request.consumer,
                inputs: request.inputs || {},
                payment: request.payment,
                status: 'received',
                receivedAt: new Date().toISOString()
            };

            // Add to queue
            await queueManager.addJob(job);

            // Send acknowledgment
            await writeMessage(stream, {
                success: true,
                jobId: job.id,
                status: 'queued',
                message: 'Job queued for execution'
            });

            console.log(`Job ${job.id} queued successfully`);
        } catch (error) {
            console.error('API request handler error:', error);
            try {
                await writeMessage(stream, {
                    success: false,
                    error: error.message
                });
            } catch (writeError) {
                console.error('Failed to send error response:', writeError);
            }
        } finally {
            try {
                await stream.close();
            } catch (closeError) {
                // Ignore close errors
            }
        }
    };
}

/**
 * Send API request to provider (consumer side)
 */
async function sendApiRequest(p2pNode, providerMultiaddr, request) {
    let stream = null;

    try {
        console.log(`Sending API request to ${providerMultiaddr}`);

        // Open stream to provider
        stream = await p2pNode.openStream(providerMultiaddr, BOB_API_PROTOCOL);

        // Send request
        await writeMessage(stream, request);

        // Read response
        const response = await readMessage(stream);

        console.log(`Received response:`, response);

        return response;
    } catch (error) {
        console.error('Failed to send API request:', error);
        throw new Error(`API request failed: ${error.message}`);
    } finally {
        if (stream) {
            try {
                await stream.close();
            } catch (closeError) {
                // Ignore close errors
            }
        }
    }
}

/**
 * Create result fetch handler for provider
 * This handles consumers fetching job results
 */
function createResultFetchHandler(queueManager) {
    const RESULT_FETCH_PROTOCOL = '/bob-result/1.0.0';

    return {
        protocol: RESULT_FETCH_PROTOCOL,
        handler: async (stream) => {
            try {
                // Read the request
                const request = await readMessage(stream);

                console.log(`Received result fetch request for job: ${request.jobId}`);

                // Validate request
                if (!request.jobId) {
                    await writeMessage(stream, {
                        success: false,
                        error: 'Invalid request: missing jobId'
                    });
                    return;
                }

                // Get job from queue
                const job = await queueManager.getJob(request.jobId);

                if (!job) {
                    await writeMessage(stream, {
                        success: false,
                        error: `Job not found: ${request.jobId}`
                    });
                    return;
                }

                // Check if job is complete
                if (job.status !== 'completed' && job.status !== 'failed') {
                    await writeMessage(stream, {
                        success: true,
                        status: job.status,
                        message: 'Job not yet complete'
                    });
                    return;
                }

                // Get result
                const result = await queueManager.getResult(request.jobId);

                // Send result
                await writeMessage(stream, {
                    success: true,
                    status: job.status,
                    result: result
                });

                console.log(`Result sent for job ${request.jobId}`);
            } catch (error) {
                console.error('Result fetch handler error:', error);
                try {
                    await writeMessage(stream, {
                        success: false,
                        error: error.message
                    });
                } catch (writeError) {
                    console.error('Failed to send error response:', writeError);
                }
            } finally {
                try {
                    await stream.close();
                } catch (closeError) {
                    // Ignore close errors
                }
            }
        }
    };
}

/**
 * Fetch result from provider (consumer side)
 */
async function fetchResult(p2pNode, providerMultiaddr, jobId) {
    const RESULT_FETCH_PROTOCOL = '/bob-result/1.0.0';
    let stream = null;

    try {
        console.log(`Fetching result for job ${jobId} from ${providerMultiaddr}`);

        // Open stream to provider
        stream = await p2pNode.openStream(providerMultiaddr, RESULT_FETCH_PROTOCOL);

        // Send request
        await writeMessage(stream, { jobId });

        // Read response
        const response = await readMessage(stream);

        console.log(`Result fetch response:`, response);

        return response;
    } catch (error) {
        console.error('Failed to fetch result:', error);
        throw new Error(`Result fetch failed: ${error.message}`);
    } finally {
        if (stream) {
            try {
                await stream.close();
            } catch (closeError) {
                // Ignore close errors
            }
        }
    }
}

module.exports = {
    BOB_API_PROTOCOL,
    createApiRequestHandler,
    sendApiRequest,
    createResultFetchHandler,
    fetchResult,
    readMessage,
    writeMessage
};
