/**
 * P2P Node using libp2p
 *
 * Handles:
 * - Peer connections
 * - NAT traversal (circuit relay, hole punching)
 * - Encrypted communications
 * - Peer discovery
 */

// Use dynamic imports for ESM packages
let libp2pModules = null;

async function loadLibp2pModules() {
    if (libp2pModules) return libp2pModules;

    const [
        { createLibp2p },
        { noise },
        { mplex },
        { tcp },
        { webSockets },
        { kadDHT },
        { bootstrap },
        { identify },
        { circuitRelayTransport },
        { multiaddr },
        { ping }
    ] = await Promise.all([
        import('libp2p'),
        import('@chainsafe/libp2p-noise'),
        import('@libp2p/mplex'),
        import('@libp2p/tcp'),
        import('@libp2p/websockets'),
        import('@libp2p/kad-dht'),
        import('@libp2p/bootstrap'),
        import('@libp2p/identify'),
        import('@libp2p/circuit-relay-v2'),
        import('@multiformats/multiaddr'),
        import('@libp2p/ping')
    ]);

    libp2pModules = {
        createLibp2p,
        noise,
        mplex,
        tcp,
        webSockets,
        kadDHT,
        bootstrap,
        identify,
        circuitRelayTransport,
        multiaddr,
        ping
    };

    return libp2pModules;
}

class P2PNode {
    constructor(config) {
        this.config = config;
        this.libp2p = null;
        this.bootstrapPeers = config.p2p?.bootstrap || [];
        this.protocols = new Map();
        this.multiaddr = null;
    }

    /**
     * Start the P2P node
     */
    async start() {
        console.log('Starting P2P node...');

        // Load libp2p modules dynamically
        const {
            createLibp2p,
            noise,
            mplex,
            tcp,
            webSockets,
            kadDHT,
            bootstrap,
            identify,
            circuitRelayTransport,
            multiaddr,
            ping
        } = await loadLibp2pModules();

        // Store multiaddr function for later use
        this.multiaddr = multiaddr;

        const libp2pConfig = {
            addresses: {
                listen: [
                    `/ip4/0.0.0.0/tcp/${this.config.p2p?.port || 0}`,
                    `/ip4/0.0.0.0/tcp/${this.config.p2p?.wsPort || 0}/ws`
                ]
            },
            transports: [
                tcp(),
                webSockets(),
                circuitRelayTransport({
                    discoverRelays: 1
                })
            ],
            connectionEncrypters: [
                noise()
            ],
            streamMuxers: [
                mplex()
            ],
            services: {
                identify: identify(),
                ping: ping(),
                dht: kadDHT({
                    clientMode: false
                })
            }
        };

        // Add bootstrap nodes if configured
        if (this.bootstrapPeers.length > 0) {
            libp2pConfig.peerDiscovery = [
                bootstrap({
                    list: this.bootstrapPeers
                })
            ];
        }

        // Create libp2p node
        this.libp2p = await createLibp2p(libp2pConfig);

        // Start the node
        await this.libp2p.start();

        const peerId = this.libp2p.peerId.toString();
        const addrs = this.libp2p.getMultiaddrs();

        console.log('P2P node started!');
        console.log(`Peer ID: ${peerId}`);
        console.log('Listening on:');
        addrs.forEach(addr => console.log(`  ${addr.toString()}`));

        // Listen for peer discovery
        this.libp2p.addEventListener('peer:discovery', (evt) => {
            const peer = evt.detail;
            console.log(`Discovered peer: ${peer.id.toString()}`);
        });

        // Listen for peer connections
        this.libp2p.addEventListener('peer:connect', (evt) => {
            const connection = evt.detail;
            console.log(`Connected to peer: ${connection.remotePeer.toString()}`);
        });

        // Listen for peer disconnections
        this.libp2p.addEventListener('peer:disconnect', (evt) => {
            const connection = evt.detail;
            console.log(`Disconnected from peer: ${connection.remotePeer.toString()}`);
        });

        return this.libp2p;
    }

    /**
     * Stop the P2P node
     */
    async stop() {
        if (this.libp2p) {
            console.log('Stopping P2P node...');
            await this.libp2p.stop();
            console.log('P2P node stopped');
        }
    }

    /**
     * Get the peer ID
     */
    getPeerId() {
        return this.libp2p?.peerId?.toString();
    }

    /**
     * Get all multiaddresses for this node
     */
    getMultiaddrs() {
        if (!this.libp2p) {
            return [];
        }
        return this.libp2p.getMultiaddrs().map(addr => addr.toString());
    }

    /**
     * Get public multiaddresses (excluding localhost and private IPs)
     * Filters out:
     * - IPv4: 127.0.0.0/8 (loopback), 10.0.0.0/8, 172.16.0.0/12, 192.168.0.0/16, 169.254.0.0/16 (link-local)
     * - IPv6: ::1 (loopback), fe80::/10 (link-local), fc00::/7 (unique local)
     */
    getPublicMultiaddrs() {
        const addrs = this.getMultiaddrs();
        return addrs.filter(addr => !this.isPrivateOrLocalAddress(addr));
    }

    /**
     * Check if a multiaddr contains a private or local address
     * @param {string} addr - Multiaddr string
     * @returns {boolean} - True if private/local
     */
    isPrivateOrLocalAddress(addr) {
        const patterns = [
            // IPv4 private/local ranges
            /\/ip4\/127\./,                          // 127.0.0.0/8 loopback
            /\/ip4\/10\./,                           // 10.0.0.0/8 private
            /\/ip4\/192\.168\./,                     // 192.168.0.0/16 private
            /\/ip4\/172\.(1[6-9]|2[0-9]|3[0-1])\./,  // 172.16.0.0/12 private (172.16-31.x.x)
            /\/ip4\/169\.254\./,                     // 169.254.0.0/16 link-local
            /\/ip4\/0\.0\.0\.0/,                     // Unspecified
            // IPv6 private/local ranges
            /\/ip6\/::1/,                            // ::1 loopback
            /\/ip6\/fe80:/i,                         // fe80::/10 link-local
            /\/ip6\/f[cd][0-9a-f]{2}:/i,             // fc00::/7 unique local (fc00-fdff)
            // Hostname
            /localhost/i,
        ];
        return patterns.some(pattern => pattern.test(addr));
    }

    /**
     * Dial a peer by multiaddr
     */
    async dial(peerMultiaddr) {
        if (!this.libp2p) {
            throw new Error('P2P node not started');
        }

        console.log(`Dialing peer: ${peerMultiaddr}`);
        const ma = this.multiaddr(peerMultiaddr);
        const connection = await this.libp2p.dial(ma);
        console.log(`Connected to: ${connection.remotePeer.toString()}`);
        return connection;
    }

    /**
     * Register a protocol handler
     * @param {string} protocolId - Protocol identifier (e.g., '/bob-api/1.0.0')
     * @param {Function} handler - Handler function (stream) => {}
     */
    async registerProtocol(protocolId, handler) {
        if (!this.libp2p) {
            throw new Error('P2P node not started');
        }

        console.log(`Registering protocol: ${protocolId}`);

        await this.libp2p.handle(protocolId, async ({ stream }) => {
            try {
                await handler(stream);
            } catch (error) {
                console.error(`Protocol handler error for ${protocolId}:`, error);
                stream.abort(error);
            }
        });

        this.protocols.set(protocolId, handler);
    }

    /**
     * Unregister a protocol handler
     */
    async unregisterProtocol(protocolId) {
        if (!this.libp2p) {
            return;
        }

        console.log(`Unregistering protocol: ${protocolId}`);
        await this.libp2p.unhandle(protocolId);
        this.protocols.delete(protocolId);
    }

    /**
     * Open a stream to a peer with a specific protocol
     */
    async openStream(peerMultiaddr, protocolId) {
        if (!this.libp2p) {
            throw new Error('P2P node not started');
        }

        // Dial the peer if not already connected
        const ma = this.multiaddr(peerMultiaddr);
        const connection = await this.libp2p.dial(ma);

        // Open a stream using the protocol
        const stream = await connection.newStream(protocolId);
        return stream;
    }

    /**
     * Get peer info
     */
    async getPeerInfo(peerId) {
        if (!this.libp2p) {
            throw new Error('P2P node not started');
        }

        const peer = await this.libp2p.peerStore.get(peerId);
        return {
            id: peer.id.toString(),
            addresses: peer.addresses.map(addr => addr.multiaddr.toString()),
            protocols: peer.protocols
        };
    }

    /**
     * Get all connected peers
     */
    getConnectedPeers() {
        if (!this.libp2p) {
            return [];
        }

        const connections = this.libp2p.getConnections();
        return connections.map(conn => ({
            peerId: conn.remotePeer.toString(),
            remoteAddr: conn.remoteAddr.toString(),
            status: conn.status
        }));
    }

    /**
     * Check if connected to a peer
     */
    isConnected(peerId) {
        if (!this.libp2p) {
            return false;
        }

        const connections = this.libp2p.getConnections(peerId);
        return connections.length > 0;
    }

    /**
     * Wait for a peer to be discovered
     */
    waitForPeerDiscovery(timeout = 30000) {
        return new Promise((resolve, reject) => {
            const timer = setTimeout(() => {
                this.libp2p.removeEventListener('peer:discovery', handler);
                reject(new Error('Peer discovery timeout'));
            }, timeout);

            const handler = (evt) => {
                clearTimeout(timer);
                this.libp2p.removeEventListener('peer:discovery', handler);
                resolve(evt.detail);
            };

            this.libp2p.addEventListener('peer:discovery', handler);
        });
    }
}

module.exports = P2PNode;
