/**
 * Job Execution System
 *
 * Handles asynchronous job execution with progress tracking
 */

const fs = require('fs').promises;
const path = require('path');
const Ajv = require('ajv');
const { generateJobId } = require('../utils/helpers');

const ajv = new Ajv();

class JobExecutor {
    constructor(database, config) {
        this.database = database;
        this.config = config;
        this.handlers = new Map(); // apiId -> handler function
        this.resultStorage = config.provider.results.storagePath;
        this.publicEndpoint = config.provider.publicEndpoint;

        // Ensure result storage directory exists
        this.initStorage();

        // Start job processor
        this.startProcessor();
    }

    async initStorage() {
        try {
            await fs.mkdir(this.resultStorage, { recursive: true });
            console.log(`Result storage initialized: ${this.resultStorage}`);
        } catch (error) {
            console.error('Failed to initialize result storage:', error);
        }
    }

    /**
     * Register API handler
     *
     * @param {object} api - API definition
     */
    async registerHandler(api) {
        try {
            // Load handler module
            const handlerPath = path.resolve(api.handler);
            const handler = require(handlerPath);

            if (typeof handler !== 'function') {
                throw new Error(`Handler ${api.handler} must export a function`);
            }

            // Compile request schema validator
            const requestValidator = ajv.compile(api.schema.request);

            // Store handler with metadata
            this.handlers.set(api.id, {
                handler,
                api,
                requestValidator
            });

            console.log(`Handler registered: ${api.id} -> ${api.handler}`);

        } catch (error) {
            console.error(`Failed to register handler for ${api.id}:`, error);
            throw error;
        }
    }

    /**
     * Create a new job
     *
     * @param {string} apiId - API identifier
     * @param {object} params - Request parameters
     * @param {string} walletAddress - Consumer wallet address
     * @param {string} transactionSignature - Payment transaction signature
     * @returns {Promise<string>} - Job ID
     */
    async createJob(apiId, params, walletAddress, transactionSignature) {
        const handlerData = this.handlers.get(apiId);

        if (!handlerData) {
            throw new Error(`No handler registered for API: ${apiId}`);
        }

        // Validate parameters
        const valid = handlerData.requestValidator(params);
        if (!valid) {
            const errors = handlerData.requestValidator.errors;
            throw new Error(`Invalid parameters: ${JSON.stringify(errors)}`);
        }

        // Create job
        const jobId = generateJobId();
        const job = {
            jobId,
            apiId,
            walletAddress,
            transactionSignature,
            params,
            status: 'queued',
            progress: 0,
            progressMessage: 'Queued',
            createdAt: Date.now()
        };

        await this.database.createJob(job);

        console.log(`Job created: ${jobId} (${apiId})`);

        return jobId;
    }

    /**
     * Get job status
     *
     * @param {string} jobId - Job identifier
     * @returns {Promise<object>} - Job data
     */
    async getJob(jobId) {
        const job = await this.database.getJob(jobId);

        if (!job) {
            throw new Error(`Job not found: ${jobId}`);
        }

        return job;
    }

    /**
     * Update job progress
     *
     * @param {string} jobId - Job identifier
     * @param {number} progress - Progress percentage (0-100)
     * @param {string} message - Progress message
     */
    async updateProgress(jobId, progress, message) {
        await this.database.updateJob(jobId, {
            progress: Math.min(100, Math.max(0, progress)),
            progressMessage: message
        });

        console.log(`Job ${jobId}: ${progress}% - ${message}`);
    }

    /**
     * Save result file
     *
     * @param {string} jobId - Job identifier
     * @param {string} sourcePath - Path to result file
     * @param {string} filename - Result filename
     * @returns {Promise<string>} - Saved filename with jobId prefix
     */
    async saveResult(jobId, sourcePath, filename) {
        const savedFilename = `${jobId}_${filename}`;
        const destPath = path.join(this.resultStorage, savedFilename);

        await fs.copyFile(sourcePath, destPath);

        // Store filename in job for P2P streaming
        await this.database.updateJob(jobId, {
            resultFilename: savedFilename
        });

        console.log(`Result saved: ${destPath}`);

        return savedFilename;
    }

    /**
     * Execute job (called by processor)
     */
    async executeJob(jobId) {
        const job = await this.database.getJob(jobId);

        if (!job) {
            console.error(`Job not found: ${jobId}`);
            return;
        }

        const handlerData = this.handlers.get(job.apiId);

        if (!handlerData) {
            console.error(`No handler for API: ${job.apiId}`);
            await this.database.updateJob(jobId, {
                status: 'failed',
                error: 'Handler not found',
                completedAt: Date.now()
            });
            await this.database.recordApiError(job.apiId);
            return;
        }

        try {
            // Update status to processing
            await this.database.updateJob(jobId, {
                status: 'processing',
                startedAt: Date.now()
            });

            console.log(`Executing job: ${jobId} (${job.apiId})`);

            // Create execution context
            const context = {
                jobId,
                updateProgress: (progress, message) => {
                    return this.updateProgress(jobId, progress, message);
                },
                saveResult: (sourcePath, filename) => {
                    return this.saveResult(jobId, sourcePath, filename);
                }
            };

            // Execute handler with timeout
            const maxDuration = handlerData.api.execution.maxDuration * 1000;
            const result = await this.executeWithTimeout(
                handlerData.handler(job.params, context),
                maxDuration
            );

            // Update job with result
            await this.database.updateJob(jobId, {
                status: 'completed',
                progress: 100,
                progressMessage: 'Completed',
                result,
                completedAt: Date.now()
            });

            await this.database.recordApiCall(job.apiId);

            console.log(`Job completed: ${jobId}`);

        } catch (error) {
            console.error(`Job failed: ${jobId}`, error);

            await this.database.updateJob(jobId, {
                status: 'failed',
                error: error.message,
                completedAt: Date.now()
            });

            await this.database.recordApiError(job.apiId);
        }
    }

    /**
     * Execute with timeout
     */
    async executeWithTimeout(promise, timeoutMs) {
        return Promise.race([
            promise,
            new Promise((_, reject) => {
                setTimeout(() => {
                    reject(new Error(`Execution timeout after ${timeoutMs}ms`));
                }, timeoutMs);
            })
        ]);
    }

    /**
     * Job processor - runs queued jobs
     */
    async processJobs() {
        try {
            // Get all queued jobs
            const queuedJobs = await this.database.getJobsByStatus('queued', 100);

            if (queuedJobs.length === 0) {
                return;
            }

            // Get currently processing jobs by API
            const processingJobs = await this.database.getJobsByStatus('processing', 100);
            const processingByApi = {};

            for (const job of processingJobs) {
                processingByApi[job.apiId] = (processingByApi[job.apiId] || 0) + 1;
            }

            // Process jobs that have capacity
            for (const job of queuedJobs) {
                const handlerData = this.handlers.get(job.apiId);

                if (!handlerData) {
                    continue;
                }

                const concurrent = handlerData.api.capacity.concurrent;
                const current = processingByApi[job.apiId] || 0;

                if (current < concurrent) {
                    // Execute job (fire and forget)
                    this.executeJob(job.jobId).catch(err => {
                        console.error(`Job execution error:`, err);
                    });

                    // Update count
                    processingByApi[job.apiId] = current + 1;
                }
            }

        } catch (error) {
            console.error('Job processor error:', error);
        }
    }

    /**
     * Start job processor
     */
    startProcessor() {
        setInterval(() => {
            this.processJobs().catch(err => {
                console.error('Job processor error:', err);
            });
        }, 1000); // Every second

        console.log('Job processor started');
    }

    /**
     * Cleanup old results
     */
    async cleanupResults() {
        try {
            const jobs = await this.database.getJobsByStatus('completed', 1000);
            const now = Date.now();

            for (const job of jobs) {
                const handlerData = this.handlers.get(job.apiId);

                if (!handlerData) {
                    continue;
                }

                const retention = handlerData.api.execution.resultRetention * 1000;
                const age = now - job.completedAt;

                if (age > retention && job.resultUrl) {
                    // Delete result file
                    const filename = job.resultUrl.split('/').pop();
                    const filepath = path.join(this.resultStorage, filename);

                    try {
                        await fs.unlink(filepath);
                        console.log(`Deleted old result: ${filename}`);

                        // Update job to remove result URL
                        await this.database.updateJob(job.jobId, {
                            resultUrl: null
                        });
                    } catch (error) {
                        // Ignore if file doesn't exist
                    }
                }
            }

        } catch (error) {
            console.error('Result cleanup error:', error);
        }
    }
}

module.exports = JobExecutor;
