/**
 * SQLite Database Implementation
 */

const Database = require('better-sqlite3');
const fs = require('fs');
const path = require('path');

class SqliteDatabase {
    constructor(config) {
        this.config = config;
        this.db = null;
    }

    async connect() {
        // Ensure directory exists
        const dir = path.dirname(this.config.path);
        if (!fs.existsSync(dir)) {
            fs.mkdirSync(dir, { recursive: true });
        }

        this.db = new Database(this.config.path);
        this.db.pragma('journal_mode = WAL');
    }

    async close() {
        if (this.db) {
            this.db.close();
        }
    }

    async init() {
        // Create tables
        this.db.exec(`
            CREATE TABLE IF NOT EXISTS queue_codes (
                code TEXT PRIMARY KEY,
                api_id TEXT NOT NULL,
                wallet_address TEXT NOT NULL,
                price REAL NOT NULL,
                position INTEGER NOT NULL,
                created_at INTEGER NOT NULL,
                expires_at INTEGER NOT NULL
            );

            CREATE INDEX IF NOT EXISTS idx_queue_expires ON queue_codes(expires_at);
            CREATE INDEX IF NOT EXISTS idx_queue_api ON queue_codes(api_id);

            CREATE TABLE IF NOT EXISTS jobs (
                job_id TEXT PRIMARY KEY,
                api_id TEXT NOT NULL,
                wallet_address TEXT NOT NULL,
                transaction_signature TEXT NOT NULL UNIQUE,
                params TEXT NOT NULL,
                status TEXT NOT NULL,
                progress INTEGER DEFAULT 0,
                progress_message TEXT,
                result TEXT,
                result_url TEXT,
                result_filename TEXT,
                error TEXT,
                created_at INTEGER NOT NULL,
                started_at INTEGER,
                completed_at INTEGER
            );

            CREATE INDEX IF NOT EXISTS idx_jobs_api ON jobs(api_id);
            CREATE INDEX IF NOT EXISTS idx_jobs_status ON jobs(status);
            CREATE INDEX IF NOT EXISTS idx_jobs_wallet ON jobs(wallet_address);
            CREATE INDEX IF NOT EXISTS idx_jobs_created ON jobs(created_at);

            CREATE TABLE IF NOT EXISTS used_transactions (
                signature TEXT PRIMARY KEY,
                job_id TEXT NOT NULL,
                amount REAL NOT NULL,
                used_at INTEGER NOT NULL
            );

            CREATE INDEX IF NOT EXISTS idx_tx_used_at ON used_transactions(used_at);

            CREATE TABLE IF NOT EXISTS earnings (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                api_id TEXT NOT NULL,
                job_id TEXT NOT NULL,
                amount REAL NOT NULL,
                wallet_address TEXT NOT NULL,
                transaction_signature TEXT NOT NULL,
                earned_at INTEGER NOT NULL
            );

            CREATE INDEX IF NOT EXISTS idx_earnings_api ON earnings(api_id);
            CREATE INDEX IF NOT EXISTS idx_earnings_date ON earnings(earned_at);

            CREATE TABLE IF NOT EXISTS api_stats (
                api_id TEXT PRIMARY KEY,
                total_calls INTEGER DEFAULT 0,
                successful_calls INTEGER DEFAULT 0,
                failed_calls INTEGER DEFAULT 0,
                total_earned REAL DEFAULT 0,
                last_call_at INTEGER,
                created_at INTEGER NOT NULL
            );
        `);

        // Migration: Add result_filename column if it doesn't exist
        try {
            this.db.exec(`
                ALTER TABLE jobs ADD COLUMN result_filename TEXT;
            `);
            console.log('Migration: Added result_filename column to jobs table');
        } catch (error) {
            // Column already exists, ignore error
            if (!error.message.includes('duplicate column name')) {
                throw error;
            }
        }
    }

    // Queue management
    createQueueCode(queueCode) {
        const stmt = this.db.prepare(`
            INSERT INTO queue_codes (code, api_id, wallet_address, price, position, created_at, expires_at)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        `);

        stmt.run(
            queueCode.code,
            queueCode.apiId,
            queueCode.walletAddress,
            queueCode.price,
            queueCode.position,
            queueCode.createdAt,
            queueCode.expiresAt
        );

        return queueCode;
    }

    getQueueCode(code) {
        const stmt = this.db.prepare('SELECT * FROM queue_codes WHERE code = ?');
        const row = stmt.get(code);

        if (!row) return null;

        return {
            code: row.code,
            apiId: row.api_id,
            walletAddress: row.wallet_address,
            price: row.price,
            position: row.position,
            createdAt: row.created_at,
            expiresAt: row.expires_at
        };
    }

    deleteQueueCode(code) {
        const stmt = this.db.prepare('DELETE FROM queue_codes WHERE code = ?');
        stmt.run(code);
    }

    cleanupExpiredQueueCodes() {
        const now = Date.now();
        const stmt = this.db.prepare('DELETE FROM queue_codes WHERE expires_at < ?');
        const result = stmt.run(now);
        return result.changes;
    }

    getQueuePosition(apiId) {
        const stmt = this.db.prepare('SELECT COUNT(*) as count FROM queue_codes WHERE api_id = ?');
        const row = stmt.get(apiId);
        return row.count;
    }

    // Job management
    createJob(job) {
        const stmt = this.db.prepare(`
            INSERT INTO jobs (
                job_id, api_id, wallet_address, transaction_signature,
                params, status, progress, progress_message, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        `);

        stmt.run(
            job.jobId,
            job.apiId,
            job.walletAddress,
            job.transactionSignature,
            JSON.stringify(job.params),
            job.status,
            job.progress || 0,
            job.progressMessage || '',
            job.createdAt
        );

        return job;
    }

    getJob(jobId) {
        const stmt = this.db.prepare('SELECT * FROM jobs WHERE job_id = ?');
        const row = stmt.get(jobId);

        if (!row) return null;

        return {
            jobId: row.job_id,
            apiId: row.api_id,
            walletAddress: row.wallet_address,
            transactionSignature: row.transaction_signature,
            params: JSON.parse(row.params),
            status: row.status,
            progress: row.progress,
            progressMessage: row.progress_message,
            result: row.result ? JSON.parse(row.result) : null,
            resultUrl: row.result_url,
            resultFilename: row.result_filename,
            error: row.error,
            createdAt: row.created_at,
            startedAt: row.started_at,
            completedAt: row.completed_at
        };
    }

    updateJob(jobId, updates) {
        const fields = [];
        const values = [];

        if (updates.status !== undefined) {
            fields.push('status = ?');
            values.push(updates.status);
        }
        if (updates.progress !== undefined) {
            fields.push('progress = ?');
            values.push(updates.progress);
        }
        if (updates.progressMessage !== undefined) {
            fields.push('progress_message = ?');
            values.push(updates.progressMessage);
        }
        if (updates.result !== undefined) {
            fields.push('result = ?');
            values.push(JSON.stringify(updates.result));
        }
        if (updates.resultUrl !== undefined) {
            fields.push('result_url = ?');
            values.push(updates.resultUrl);
        }
        if (updates.resultFilename !== undefined) {
            fields.push('result_filename = ?');
            values.push(updates.resultFilename);
        }
        if (updates.error !== undefined) {
            fields.push('error = ?');
            values.push(updates.error);
        }
        if (updates.startedAt !== undefined) {
            fields.push('started_at = ?');
            values.push(updates.startedAt);
        }
        if (updates.completedAt !== undefined) {
            fields.push('completed_at = ?');
            values.push(updates.completedAt);
        }

        if (fields.length === 0) return;

        values.push(jobId);

        const stmt = this.db.prepare(`
            UPDATE jobs SET ${fields.join(', ')} WHERE job_id = ?
        `);

        stmt.run(...values);
    }

    getJobsByApi(apiId, limit = 100) {
        const stmt = this.db.prepare(`
            SELECT * FROM jobs
            WHERE api_id = ?
            ORDER BY created_at DESC
            LIMIT ?
        `);

        const rows = stmt.all(apiId, limit);

        return rows.map(row => ({
            jobId: row.job_id,
            apiId: row.api_id,
            walletAddress: row.wallet_address,
            transactionSignature: row.transaction_signature,
            params: JSON.parse(row.params),
            status: row.status,
            progress: row.progress,
            progressMessage: row.progress_message,
            result: row.result ? JSON.parse(row.result) : null,
            resultUrl: row.result_url,
            error: row.error,
            createdAt: row.created_at,
            startedAt: row.started_at,
            completedAt: row.completed_at
        }));
    }

    getJobsByStatus(status, limit = 100) {
        const stmt = this.db.prepare(`
            SELECT * FROM jobs
            WHERE status = ?
            ORDER BY created_at ASC
            LIMIT ?
        `);

        const rows = stmt.all(status, limit);

        return rows.map(row => ({
            jobId: row.job_id,
            apiId: row.api_id,
            walletAddress: row.wallet_address,
            transactionSignature: row.transaction_signature,
            params: JSON.parse(row.params),
            status: row.status,
            progress: row.progress,
            progressMessage: row.progress_message,
            result: row.result ? JSON.parse(row.result) : null,
            resultUrl: row.result_url,
            error: row.error,
            createdAt: row.created_at,
            startedAt: row.started_at,
            completedAt: row.completed_at
        }));
    }

    // Transaction tracking
    isTransactionUsed(signature) {
        const stmt = this.db.prepare('SELECT 1 FROM used_transactions WHERE signature = ?');
        const row = stmt.get(signature);
        return row !== undefined;
    }

    markTransactionUsed(signature, jobId, amount) {
        const stmt = this.db.prepare(`
            INSERT INTO used_transactions (signature, job_id, amount, used_at)
            VALUES (?, ?, ?, ?)
        `);

        stmt.run(signature, jobId, amount, Date.now());
    }

    // Earnings tracking
    recordEarning(earning) {
        const stmt = this.db.prepare(`
            INSERT INTO earnings (api_id, job_id, amount, wallet_address, transaction_signature, earned_at)
            VALUES (?, ?, ?, ?, ?, ?)
        `);

        stmt.run(
            earning.apiId,
            earning.jobId,
            earning.amount,
            earning.walletAddress,
            earning.transactionSignature,
            earning.earnedAt
        );

        // Update API stats
        const updateStmt = this.db.prepare(`
            UPDATE api_stats
            SET total_earned = total_earned + ?
            WHERE api_id = ?
        `);

        updateStmt.run(earning.amount, earning.apiId);
    }

    getEarnings(startDate, endDate) {
        const stmt = this.db.prepare(`
            SELECT * FROM earnings
            WHERE earned_at >= ? AND earned_at <= ?
            ORDER BY earned_at DESC
        `);

        const rows = stmt.all(startDate, endDate);

        return rows.map(row => ({
            id: row.id,
            apiId: row.api_id,
            jobId: row.job_id,
            amount: row.amount,
            walletAddress: row.wallet_address,
            transactionSignature: row.transaction_signature,
            earnedAt: row.earned_at
        }));
    }

    getTotalEarnings() {
        const stmt = this.db.prepare('SELECT SUM(amount) as total FROM earnings');
        const row = stmt.get();
        return row.total || 0;
    }

    // API statistics
    recordApiCall(apiId) {
        // Create stats entry if doesn't exist
        const insertStmt = this.db.prepare(`
            INSERT OR IGNORE INTO api_stats (api_id, created_at)
            VALUES (?, ?)
        `);
        insertStmt.run(apiId, Date.now());

        // Update stats
        const updateStmt = this.db.prepare(`
            UPDATE api_stats
            SET total_calls = total_calls + 1,
                successful_calls = successful_calls + 1,
                last_call_at = ?
            WHERE api_id = ?
        `);

        updateStmt.run(Date.now(), apiId);
    }

    recordApiError(apiId) {
        // Create stats entry if doesn't exist
        const insertStmt = this.db.prepare(`
            INSERT OR IGNORE INTO api_stats (api_id, created_at)
            VALUES (?, ?)
        `);
        insertStmt.run(apiId, Date.now());

        // Update stats
        const updateStmt = this.db.prepare(`
            UPDATE api_stats
            SET total_calls = total_calls + 1,
                failed_calls = failed_calls + 1,
                last_call_at = ?
            WHERE api_id = ?
        `);

        updateStmt.run(Date.now(), apiId);
    }

    getApiStats(apiId) {
        const stmt = this.db.prepare('SELECT * FROM api_stats WHERE api_id = ?');
        const row = stmt.get(apiId);

        if (!row) return null;

        return {
            apiId: row.api_id,
            totalCalls: row.total_calls,
            successfulCalls: row.successful_calls,
            failedCalls: row.failed_calls,
            totalEarned: row.total_earned,
            lastCallAt: row.last_call_at,
            createdAt: row.created_at
        };
    }

    getAllApiStats() {
        const stmt = this.db.prepare('SELECT * FROM api_stats ORDER BY total_calls DESC');
        const rows = stmt.all();

        return rows.map(row => ({
            apiId: row.api_id,
            totalCalls: row.total_calls,
            successfulCalls: row.successful_calls,
            failedCalls: row.failed_calls,
            totalEarned: row.total_earned,
            lastCallAt: row.last_call_at,
            createdAt: row.created_at
        }));
    }
}

module.exports = SqliteDatabase;
