const fs = require('fs');
const path = require('path');

/**
 * Simple JSON file-based database for testing
 */
class JsonDatabase {
    constructor(dbConfig) {
        this.dbPath = dbConfig.path.replace('.db', '.json');
        this.data = {
            queue_codes: [],
            jobs: [],
            used_transactions: [],
            earnings: [],
            api_stats: []
        };
        this.loadData();
    }

    loadData() {
        try {
            if (fs.existsSync(this.dbPath)) {
                const content = fs.readFileSync(this.dbPath, 'utf8');
                this.data = JSON.parse(content);
            }
        } catch (error) {
            console.warn('Could not load existing data, starting fresh');
        }
    }

    saveData() {
        fs.writeFileSync(this.dbPath, JSON.stringify(this.data, null, 2));
    }

    async connect() {
        // No-op for JSON db
    }

    async init() {
        // No-op for JSON db
    }

    async close() {
        this.saveData();
    }

    // Queue codes
    async createQueueCode(queueCode) {
        this.data.queue_codes.push(queueCode);
        this.saveData();
    }

    async getQueueCode(code) {
        return this.data.queue_codes.find(q => q.code === code);
    }

    async deleteQueueCode(code) {
        this.data.queue_codes = this.data.queue_codes.filter(q => q.code !== code);
        this.saveData();
    }

    async cleanupExpiredQueueCodes() {
        const now = new Date();
        this.data.queue_codes = this.data.queue_codes.filter(q => new Date(q.expires_at) > now);
        this.saveData();
    }

    async getQueuePosition(apiId) {
        return this.data.queue_codes.filter(q => q.api_id === apiId).length;
    }

    // Jobs
    async createJob(job) {
        job.created_at = new Date().toISOString();
        this.data.jobs.push(job);
        this.saveData();
        return job.job_id;
    }

    async getJob(jobId) {
        return this.data.jobs.find(j => j.job_id === jobId);
    }

    async updateJob(jobId, updates) {
        const job = this.data.jobs.find(j => j.job_id === jobId);
        if (job) {
            Object.assign(job, updates, { updated_at: new Date().toISOString() });
            this.saveData();
        }
    }

    async getJobsByApi(apiId, limit = 100) {
        return this.data.jobs
            .filter(j => j.api_id === apiId)
            .slice(-limit);
    }

    async getJobsByStatus(status, limit = 100) {
        return this.data.jobs
            .filter(j => j.status === status)
            .slice(-limit);
    }

    // Transactions
    async isTransactionUsed(signature) {
        return this.data.used_transactions.some(t => t.signature === signature);
    }

    async markTransactionUsed(signature, jobId, amount) {
        this.data.used_transactions.push({
            signature,
            job_id: jobId,
            amount,
            used_at: new Date().toISOString()
        });
        this.saveData();
    }

    // Earnings
    async recordEarning(earning) {
        earning.earned_at = new Date().toISOString();
        this.data.earnings.push(earning);
        this.saveData();
    }

    async getEarnings(startDate, endDate) {
        return this.data.earnings.filter(e => {
            const date = new Date(e.earned_at);
            return date >= startDate && date <= endDate;
        });
    }

    async getTotalEarnings() {
        return this.data.earnings.reduce((sum, e) => sum + e.amount, 0);
    }

    // API stats
    async recordApiCall(apiId) {
        let stats = this.data.api_stats.find(s => s.api_id === apiId);
        if (!stats) {
            stats = { api_id: apiId, total_calls: 0, successful_calls: 0, failed_calls: 0, total_earned: 0 };
            this.data.api_stats.push(stats);
        }
        stats.total_calls++;
        stats.successful_calls++;
        stats.last_call_at = new Date().toISOString();
        this.saveData();
    }

    async recordApiError(apiId) {
        let stats = this.data.api_stats.find(s => s.api_id === apiId);
        if (!stats) {
            stats = { api_id: apiId, total_calls: 0, successful_calls: 0, failed_calls: 0, total_earned: 0 };
            this.data.api_stats.push(stats);
        }
        stats.total_calls++;
        stats.failed_calls++;
        stats.last_call_at = new Date().toISOString();
        this.saveData();
    }

    async getApiStats(apiId) {
        return this.data.api_stats.find(s => s.api_id === apiId) || {
            api_id: apiId,
            total_calls: 0,
            successful_calls: 0,
            failed_calls: 0,
            total_earned: 0
        };
    }

    async getAllApiStats() {
        return this.data.api_stats;
    }
}

module.exports = JsonDatabase;
