/**
 * Database Module - Abstraction layer for multiple database backends
 *
 * Supports: SQLite, PostgreSQL, MongoDB, MS SQL
 */

const SqliteDatabase = require('./sqlite');
const PostgresDatabase = require('./postgres');
const MongoDatabase = require('./mongo');
const MsSqlDatabase = require('./mssql');

class Database {
    constructor(config) {
        this.config = config;
        this.db = null;
    }

    async connect() {
        // Support both config.database and config.provider.database
        const dbConfig = this.config.database || this.config.provider?.database;
        if (!dbConfig) {
            throw new Error('Database configuration not found');
        }
        const { type } = dbConfig;

        switch (type) {
            case 'sqlite':
                this.db = new SqliteDatabase(dbConfig);
                break;
            case 'postgres':
            case 'postgresql':
                this.db = new PostgresDatabase(dbConfig);
                break;
            case 'mongodb':
            case 'mongo':
                this.db = new MongoDatabase(dbConfig);
                break;
            case 'mssql':
                this.db = new MsSqlDatabase(dbConfig);
                break;
            default:
                throw new Error(`Unsupported database type: ${type}`);
        }

        await this.db.connect();
        await this.db.init();

        console.log(`Database connected: ${type}`);
        return this;
    }

    async close() {
        if (this.db) {
            await this.db.close();
        }
    }

    // Queue management
    async createQueueCode(queueCode) {
        return this.db.createQueueCode(queueCode);
    }

    async getQueueCode(code) {
        return this.db.getQueueCode(code);
    }

    async deleteQueueCode(code) {
        return this.db.deleteQueueCode(code);
    }

    async cleanupExpiredQueueCodes() {
        return this.db.cleanupExpiredQueueCodes();
    }

    async getQueuePosition(apiId) {
        return this.db.getQueuePosition(apiId);
    }

    // Job management
    async createJob(job) {
        return this.db.createJob(job);
    }

    async getJob(jobId) {
        return this.db.getJob(jobId);
    }

    async updateJob(jobId, updates) {
        return this.db.updateJob(jobId, updates);
    }

    async getJobsByApi(apiId, limit = 100) {
        return this.db.getJobsByApi(apiId, limit);
    }

    async getJobsByStatus(status, limit = 100) {
        return this.db.getJobsByStatus(status, limit);
    }

    // Transaction tracking
    async isTransactionUsed(signature) {
        return this.db.isTransactionUsed(signature);
    }

    async markTransactionUsed(signature, jobId, amount) {
        return this.db.markTransactionUsed(signature, jobId, amount);
    }

    // Earnings tracking
    async recordEarning(earning) {
        return this.db.recordEarning(earning);
    }

    async getEarnings(startDate, endDate) {
        return this.db.getEarnings(startDate, endDate);
    }

    async getTotalEarnings() {
        return this.db.getTotalEarnings();
    }

    // API statistics
    async recordApiCall(apiId) {
        return this.db.recordApiCall(apiId);
    }

    async recordApiError(apiId) {
        return this.db.recordApiError(apiId);
    }

    async getApiStats(apiId) {
        return this.db.getApiStats(apiId);
    }

    async getAllApiStats() {
        return this.db.getAllApiStats();
    }
}

module.exports = Database;
