/**
 * P2P Consumer Module
 *
 * Handles API calls over P2P network
 */

const P2PNode = require('../p2p/node');
const { sendApiRequest, fetchResult } = require('../p2p/protocols');
const { v4: uuidv4 } = require('uuid');

class P2PConsumer {
    constructor(config, solanaClient) {
        this.config = config;
        this.solana = solanaClient;
        this.p2pNode = null;
    }

    /**
     * Start the P2P consumer node
     */
    async start() {
        if (this.p2pNode) {
            return; // Already started
        }

        console.log('Starting P2P consumer node...');
        this.p2pNode = new P2PNode(this.config);
        await this.p2pNode.start();
        console.log('P2P consumer node started!');
    }

    /**
     * Stop the P2P consumer node
     */
    async stop() {
        if (this.p2pNode) {
            await this.p2pNode.stop();
            this.p2pNode = null;
        }
    }

    /**
     * Check if an endpoint is a P2P multiaddr
     */
    isP2PEndpoint(endpoint) {
        return endpoint && (
            endpoint.startsWith('/ip4/') ||
            endpoint.startsWith('/ip6/') ||
            endpoint.startsWith('/dns/') ||
            endpoint.startsWith('/p2p/')
        );
    }

    /**
     * Send payment to provider
     */
    async sendPayment(providerWallet, amount) {
        return await this.solana.sendPayment(providerWallet, amount);
    }

    /**
     * Execute API call over P2P
     *
     * @param {string} providerMultiaddr - Provider P2P multiaddr
     * @param {string} providerWallet - Provider wallet address
     * @param {string} apiId - API identifier
     * @param {number} price - API price
     * @param {object} inputs - API input parameters
     * @returns {Promise<object>} - Job response
     */
    async executeApi(providerMultiaddr, providerWallet, apiId, price, inputs) {
        if (!this.p2pNode) {
            await this.start();
        }

        console.log(`\n=== Calling API via P2P: ${apiId} ===\n`);

        // Step 1: Send payment
        console.log('Step 1: Sending payment...');
        console.log(`Amount: ${price} ${this.config.token.symbol}`);
        const transactionSignature = await this.sendPayment(providerWallet, price);
        console.log(`Transaction: ${transactionSignature}\n`);

        // Wait for confirmations
        console.log('Waiting for transaction confirmations...');
        await new Promise(resolve => setTimeout(resolve, 5000));
        console.log('Confirmations received\n');

        // Step 2: Send API request over P2P
        console.log('Step 2: Sending API request over P2P...');

        const jobId = uuidv4();
        const request = {
            jobId,
            apiId,
            consumer: this.config.wallet.address,
            inputs: inputs || {},
            payment: {
                signature: transactionSignature,
                amount: price,
                token: this.config.token.mint
            }
        };

        const response = await sendApiRequest(this.p2pNode, providerMultiaddr, request);

        if (!response.success) {
            throw new Error(`API request failed: ${response.error}`);
        }

        console.log(`Job ID: ${response.jobId}`);
        console.log(`Status: ${response.status}\n`);

        return {
            jobId: response.jobId,
            status: response.status,
            message: response.message,
            providerMultiaddr
        };
    }

    /**
     * Get job status and result over P2P
     *
     * @param {string} providerMultiaddr - Provider P2P multiaddr
     * @param {string} jobId - Job identifier
     * @returns {Promise<object>} - Job result
     */
    async getJobResult(providerMultiaddr, jobId) {
        if (!this.p2pNode) {
            await this.start();
        }

        const response = await fetchResult(this.p2pNode, providerMultiaddr, jobId);

        if (!response.success) {
            throw new Error(`Failed to fetch result: ${response.error}`);
        }

        return {
            jobId,
            status: response.status,
            result: response.result
        };
    }

    /**
     * Poll for job completion
     *
     * @param {string} providerMultiaddr - Provider P2P multiaddr
     * @param {string} jobId - Job identifier
     * @param {number} maxWaitTime - Maximum wait time in seconds
     * @param {number} pollInterval - Polling interval in seconds
     * @returns {Promise<object>} - Completed job result
     */
    async pollForCompletion(providerMultiaddr, jobId, maxWaitTime = 300, pollInterval = 2) {
        if (!this.p2pNode) {
            await this.start();
        }

        const startTime = Date.now();
        const maxWaitMs = maxWaitTime * 1000;
        const pollIntervalMs = pollInterval * 1000;

        console.log('Step 3: Waiting for completion...');

        while (true) {
            const result = await this.getJobResult(providerMultiaddr, jobId);

            if (result.status === 'completed') {
                console.log('\nJob completed!');
                return result;
            }

            if (result.status === 'failed') {
                throw new Error(`Job failed: ${result.result?.error || 'Unknown error'}`);
            }

            const elapsed = Date.now() - startTime;

            if (elapsed > maxWaitMs) {
                throw new Error(`Job timeout after ${maxWaitTime} seconds`);
            }

            if (result.status) {
                console.log(`Job ${jobId}: ${result.status}`);
            }

            await new Promise(resolve => setTimeout(resolve, pollIntervalMs));
        }
    }

    /**
     * Full API call workflow (pay -> execute -> poll)
     *
     * @param {string} providerMultiaddr - Provider P2P multiaddr
     * @param {string} providerWallet - Provider wallet address
     * @param {string} apiId - API identifier
     * @param {number} price - API price
     * @param {object} inputs - API input parameters
     * @returns {Promise<object>} - Completed job with result
     */
    async callApi(providerMultiaddr, providerWallet, apiId, price, inputs) {
        // Execute API
        const job = await this.executeApi(providerMultiaddr, providerWallet, apiId, price, inputs);

        // Poll for completion
        const result = await this.pollForCompletion(providerMultiaddr, job.jobId);

        return result;
    }
}

module.exports = P2PConsumer;
