/**
 * Consumer Module
 *
 * Provides functionality for discovering and calling APIs
 */

const axios = require('axios');
const fs = require('fs');
const path = require('path');

class Consumer {
    constructor(config, solanaClient) {
        this.config = config;
        this.solana = solanaClient;
        this.aggregators = config.aggregators || [];

        // Initialize result output directory
        if (config.consumer && config.consumer.results && config.consumer.results.outputPath) {
            this.outputPath = config.consumer.results.outputPath;
            this.initOutputDirectory();
        }
    }

    /**
     * Initialize output directory for result files
     */
    initOutputDirectory() {
        if (!fs.existsSync(this.outputPath)) {
            fs.mkdirSync(this.outputPath, { recursive: true });
            console.log(`Consumer output directory initialized: ${this.outputPath}`);
        }
    }

    /**
     * Search for APIs across all aggregators
     *
     * @param {object} filters - Search filters (category, tags, maxPrice, etc.)
     * @returns {Promise<array>} - Array of API listings
     */
    async searchApis(filters = {}) {
        const results = [];

        for (const aggregatorUrl of this.aggregators) {
            try {
                const response = await axios.get(`${aggregatorUrl}/search`, {
                    params: filters,
                    timeout: 10000
                });

                if (response.data && response.data.results) {
                    results.push(...response.data.results);
                }

            } catch (error) {
                console.error(`Search failed for ${aggregatorUrl}:`, error.message);
            }
        }

        return results;
    }

    /**
     * Get API status from provider
     *
     * @param {string} providerUrl - Provider endpoint
     * @param {string} apiId - API identifier
     * @returns {Promise<object>} - API status
     */
    async getApiStatus(providerUrl, apiId) {
        const response = await axios.get(`${providerUrl}/api/${apiId}/status`, {
            timeout: 5000
        });

        return response.data;
    }

    /**
     * Request queue position
     *
     * @param {string} providerUrl - Provider endpoint
     * @param {string} apiId - API identifier
     * @returns {Promise<object>} - Queue data (code, position, price, expiry)
     */
    async requestQueue(providerUrl, apiId) {
        const response = await axios.post(
            `${providerUrl}/api/${apiId}/queue`,
            {
                walletAddress: this.config.wallet.address
            },
            {
                timeout: 5000
            }
        );

        return response.data;
    }

    /**
     * Send payment to provider
     *
     * @param {string} providerWallet - Provider wallet address
     * @param {number} amount - Amount to pay
     * @returns {Promise<string>} - Transaction signature
     */
    async sendPayment(providerWallet, amount) {
        return await this.solana.sendPayment(providerWallet, amount);
    }

    /**
     * Execute API call
     *
     * @param {string} providerUrl - Provider endpoint
     * @param {string} apiId - API identifier
     * @param {string} queueCode - Queue code
     * @param {string} transactionSignature - Payment transaction signature
     * @param {object} params - API parameters
     * @returns {Promise<object>} - Job data
     */
    async executeApi(providerUrl, apiId, queueCode, transactionSignature, params) {
        const response = await axios.post(
            `${providerUrl}/api/${apiId}/execute`,
            {
                queueCode,
                transactionSignature,
                walletAddress: this.config.wallet.address,
                params
            },
            {
                timeout: 10000
            }
        );

        return response.data;
    }

    /**
     * Get job status
     *
     * @param {string} providerUrl - Provider endpoint
     * @param {string} jobId - Job identifier
     * @returns {Promise<object>} - Job data
     */
    async getJobStatus(providerUrl, jobId) {
        const response = await axios.get(`${providerUrl}/job/${jobId}`, {
            timeout: 5000
        });

        return response.data;
    }

    /**
     * Download result file
     *
     * @param {string} resultUrl - Result URL
     * @param {string} outputPath - Local path to save file
     */
    async downloadResult(resultUrl, outputPath) {
        const response = await axios.get(resultUrl, {
            responseType: 'stream',
            timeout: 60000
        });

        const writer = fs.createWriteStream(outputPath);

        response.data.pipe(writer);

        return new Promise((resolve, reject) => {
            writer.on('finish', resolve);
            writer.on('error', reject);
        });
    }

    /**
     * Download job result file via P2P streaming
     *
     * @param {string} providerUrl - Provider endpoint
     * @param {string} jobId - Job identifier
     * @param {string} resultFilename - Result filename
     * @returns {Promise<string>} - Local file path
     */
    async downloadJobResult(providerUrl, jobId, resultFilename) {
        const downloadUrl = `${providerUrl}/job/${jobId}/download`;
        const localPath = path.join(this.outputPath, resultFilename);

        console.log(`Downloading result from provider via P2P...`);

        await this.downloadResult(downloadUrl, localPath);

        console.log(`Result saved to: ${localPath}`);

        return localPath;
    }

    /**
     * Poll for job completion
     *
     * @param {string} providerUrl - Provider endpoint
     * @param {string} jobId - Job identifier
     * @param {number} maxWaitTime - Maximum wait time in seconds
     * @param {number} pollInterval - Polling interval in seconds
     * @returns {Promise<object>} - Completed job data
     */
    async pollForCompletion(providerUrl, jobId, maxWaitTime = 300, pollInterval = 2) {
        const startTime = Date.now();
        const maxWaitMs = maxWaitTime * 1000;
        const pollIntervalMs = pollInterval * 1000;

        while (true) {
            const job = await this.getJobStatus(providerUrl, jobId);

            if (job.status === 'completed') {
                return job;
            }

            if (job.status === 'failed') {
                throw new Error(`Job failed: ${job.error}`);
            }

            const elapsed = Date.now() - startTime;

            if (elapsed > maxWaitMs) {
                throw new Error(`Job timeout after ${maxWaitTime} seconds`);
            }

            console.log(`Job ${jobId}: ${job.progress}% - ${job.progressMessage}`);

            await new Promise(resolve => setTimeout(resolve, pollIntervalMs));
        }
    }

    /**
     * Full API call workflow (queue -> pay -> execute -> poll)
     *
     * @param {string} providerUrl - Provider endpoint
     * @param {string} providerWallet - Provider wallet address
     * @param {string} apiId - API identifier
     * @param {object} params - API parameters
     * @returns {Promise<object>} - Completed job with result
     */
    async callApi(providerUrl, providerWallet, apiId, params) {
        console.log(`\n=== Calling API: ${apiId} ===\n`);

        // Step 1: Request queue position
        console.log('Step 1: Requesting queue position...');
        const queueData = await this.requestQueue(providerUrl, apiId);
        console.log(`Queue code: ${queueData.code}`);
        console.log(`Position: ${queueData.position}`);
        console.log(`Price: ${queueData.price} ${this.config.token.symbol}`);
        console.log(`Expires in: ${queueData.expirySeconds} seconds\n`);

        // Step 2: Send payment
        console.log('Step 2: Sending payment...');
        const signature = await this.sendPayment(providerWallet, queueData.price);
        console.log(`Transaction: ${signature}\n`);

        // Wait for additional confirmations (provider requires 3 confirmations)
        console.log('Waiting for transaction confirmations...');
        await new Promise(resolve => setTimeout(resolve, 3000)); // Wait ~3 seconds for confirmations
        console.log('Confirmations received\n');

        // Step 3: Execute API
        console.log('Step 3: Executing API...');
        const job = await this.executeApi(
            providerUrl,
            apiId,
            queueData.code,
            signature,
            params
        );
        console.log(`Job ID: ${job.jobId}`);
        console.log(`Status: ${job.status}\n`);

        // Step 4: Poll for completion
        console.log('Step 4: Waiting for completion...');
        const completedJob = await this.pollForCompletion(providerUrl, job.jobId);
        console.log(`\nJob completed!`);

        // Step 5: Download result file via P2P if available
        if (completedJob.resultFilename && this.outputPath) {
            console.log('\nStep 5: Downloading result file...');
            const localFilePath = await this.downloadJobResult(
                providerUrl,
                completedJob.jobId,
                completedJob.resultFilename
            );

            // Add local file path to result
            completedJob.localFilePath = localFilePath;
        }

        return completedJob;
    }
}

module.exports = Consumer;
