#!/usr/bin/env node

/**
 * Provider CLI - View queue status
 *
 * Usage: npm run queue-status -- --config config.json
 */

const minimist = require('minimist');
const { loadConfig, loadApiDefinitions } = require('../utils/config');
const Database = require('../database');
const QueueManager = require('../queue');

async function main() {
    const args = minimist(process.argv.slice(2));

    try {
        // Load configuration
        if (!args.config) {
            console.error('Error: --config argument required');
            console.log('Usage: npm run queue-status -- --config config.json');
            process.exit(1);
        }

        const config = loadConfig(args.config);

        // Initialize database
        const database = new Database(config);
        await database.connect();

        // Initialize queue manager
        const queueManager = new QueueManager(database, config);

        console.log('\n=== Queue Status ===\n');

        // Get all API IDs from database
        const allStats = await database.getAllApiStats();

        if (allStats.length === 0) {
            console.log('No APIs registered yet.');
            await database.close();
            return;
        }

        for (const stats of allStats) {
            const queueLength = await database.getQueuePosition(stats.apiId);
            const activeJobs = await database.getJobsByStatus('processing');
            const apiActiveJobs = activeJobs.filter(job => job.apiId === stats.apiId).length;

            console.log(`API: ${stats.apiId}`);
            console.log(`  Queue Length: ${queueLength}`);
            console.log(`  Active Jobs: ${apiActiveJobs}`);
            console.log(`  Total Calls: ${stats.totalCalls}`);
            console.log(`  Success Rate: ${stats.totalCalls > 0 ? ((stats.successfulCalls / stats.totalCalls) * 100).toFixed(1) : 0}%`);
            console.log();
        }

        await database.close();

    } catch (error) {
        console.error('Error:', error.message);
        process.exit(1);
    }
}

main();
