#!/usr/bin/env node

/**
 * Provider CLI - View job history
 *
 * Usage: npm run jobs -- --config config.json [--api api-id] [--status queued|processing|completed|failed] [--limit 50]
 */

const minimist = require('minimist');
const { loadConfig } = require('../utils/config');
const Database = require('../database');

async function main() {
    const args = minimist(process.argv.slice(2));

    try {
        // Load configuration
        if (!args.config) {
            console.error('Error: --config argument required');
            console.log('Usage: npm run jobs -- --config config.json');
            process.exit(1);
        }

        const config = loadConfig(args.config);

        // Initialize database
        const database = new Database(config);
        await database.connect();

        const limit = args.limit || 50;

        console.log('\n=== Job History ===\n');

        let jobs;
        if (args.api) {
            console.log(`Filtering by API: ${args.api}\n`);
            jobs = await database.getJobsByApi(args.api, limit);
        } else if (args.status) {
            console.log(`Filtering by status: ${args.status}\n`);
            jobs = await database.getJobsByStatus(args.status, limit);
        } else {
            // Get all recent jobs (mix of statuses)
            const queued = await database.getJobsByStatus('queued', limit / 4);
            const processing = await database.getJobsByStatus('processing', limit / 4);
            const completed = await database.getJobsByStatus('completed', limit / 4);
            const failed = await database.getJobsByStatus('failed', limit / 4);
            jobs = [...queued, ...processing, ...completed, ...failed];
            jobs.sort((a, b) => b.createdAt - a.createdAt);
            jobs = jobs.slice(0, limit);
        }

        if (jobs.length === 0) {
            console.log('No jobs found.');
            await database.close();
            return;
        }

        console.log(`Showing ${jobs.length} job(s):\n`);

        for (const job of jobs) {
            const createdDate = new Date(job.createdAt).toISOString();
            console.log(`Job ID: ${job.jobId}`);
            console.log(`  API: ${job.apiId}`);
            console.log(`  Status: ${job.status}`);
            console.log(`  Progress: ${job.progress}%`);
            console.log(`  Created: ${createdDate}`);

            if (job.startedAt) {
                const duration = (job.completedAt || Date.now()) - job.startedAt;
                console.log(`  Duration: ${(duration / 1000).toFixed(1)}s`);
            }

            if (job.status === 'failed' && job.error) {
                console.log(`  Error: ${job.error}`);
            }

            console.log();
        }

        await database.close();

    } catch (error) {
        console.error('Error:', error.message);
        process.exit(1);
    }
}

main();
