#!/usr/bin/env node

/**
 * Provider CLI - View earnings report
 *
 * Usage: npm run earnings -- --config config.json [--days 30]
 */

const minimist = require('minimist');
const { loadConfig } = require('../utils/config');
const Database = require('../database');
const SolanaClient = require('../solana');
const PaymentVerifier = require('../payment');

async function main() {
    const args = minimist(process.argv.slice(2));

    try {
        // Load configuration
        if (!args.config) {
            console.error('Error: --config argument required');
            console.log('Usage: npm run earnings -- --config config.json');
            process.exit(1);
        }

        const config = loadConfig(args.config);

        // Initialize database
        const database = new Database(config);
        await database.connect();

        // Initialize payment verifier
        const solana = new SolanaClient(config);
        const paymentVerifier = new PaymentVerifier(database, solana, config);

        // Get date range
        const days = args.days || 30;
        const endDate = Date.now();
        const startDate = endDate - (days * 24 * 60 * 60 * 1000);

        console.log(`\n=== Earnings Report (Last ${days} days) ===\n`);

        // Get earnings
        const report = await paymentVerifier.getEarningsReport(startDate, endDate);

        console.log(`Total Earnings: ${report.total.toFixed(4)} ${config.token.symbol}`);
        console.log(`Total Calls: ${report.count}`);
        console.log();

        if (report.byApi.length > 0) {
            console.log('By API:');
            for (const apiStats of report.byApi) {
                console.log(`  ${apiStats.apiId}:`);
                console.log(`    Calls: ${apiStats.count}`);
                console.log(`    Earned: ${apiStats.total.toFixed(4)} ${config.token.symbol}`);
                console.log();
            }
        }

        if (report.recent.length > 0) {
            console.log('Recent Earnings:');
            for (const earning of report.recent.slice(0, 10)) {
                const date = new Date(earning.earnedAt).toISOString();
                console.log(`  ${date} - ${earning.amount.toFixed(4)} ${config.token.symbol} (${earning.apiId})`);
            }
        }

        await database.close();

    } catch (error) {
        console.error('Error:', error.message);
        process.exit(1);
    }
}

main();
