#!/usr/bin/env node

/**
 * Consumer CLI - Get API status
 *
 * Usage: npm run status <api-id> -- --config config.json --provider http://provider-url
 */

const minimist = require('minimist');
const { loadConfig } = require('../utils/config');
const SolanaClient = require('../solana');
const Consumer = require('../consumer');

async function main() {
    const args = minimist(process.argv.slice(2));

    try {
        // Get API ID
        const apiId = args._[0];
        if (!apiId) {
            console.error('Error: API ID required');
            console.log('Usage: npm run status <api-id> -- --config config.json --provider http://provider-url');
            process.exit(1);
        }

        // Load configuration
        if (!args.config) {
            console.error('Error: --config argument required');
            process.exit(1);
        }

        const config = loadConfig(args.config);

        // Get provider URL
        if (!args.provider) {
            console.error('Error: --provider argument required');
            console.log('Usage: npm run status <api-id> -- --config config.json --provider http://provider-url');
            process.exit(1);
        }

        // Initialize consumer
        const solana = new SolanaClient(config);
        const consumer = new Consumer(config, solana);

        console.log(`Getting status for ${apiId}...\n`);

        // Get status
        const status = await consumer.getApiStatus(args.provider, apiId);

        console.log(`API: ${status.name} (${status.apiId})`);
        console.log(`Description: ${status.description}`);
        console.log(`Version: ${status.version}`);
        console.log(`Price: ${status.price} ${config.token.symbol} per ${status.unit}`);
        console.log();
        console.log('Capacity:');
        console.log(`  Concurrent: ${status.capacity.concurrent}`);
        console.log(`  Queue Max: ${status.capacity.queueMax}`);
        console.log(`  Queue Timeout: ${status.capacity.queueTimeout}s`);
        console.log();
        console.log('Current Status:');
        console.log(`  Queue Length: ${status.currentQueue}`);
        console.log(`  Active Jobs: ${status.activeJobs}`);
        console.log(`  Available: ${status.available ? 'Yes' : 'No'}`);
        console.log(`  Queue Available: ${status.queueAvailable ? 'Yes' : 'No'}`);
        console.log(`  Estimated Wait: ${status.estimatedWait}s`);

    } catch (error) {
        console.error('Error:', error.message);
        process.exit(1);
    }
}

main();
