#!/usr/bin/env node

/**
 * Consumer CLI - Search for APIs
 *
 * Usage: npm run search -- --config config.json [--category ml] [--tag image-generation] [--max-price 1.0]
 */

const minimist = require('minimist');
const { loadConfig } = require('../utils/config');
const SolanaClient = require('../solana');
const Consumer = require('../consumer');

async function main() {
    const args = minimist(process.argv.slice(2));

    try {
        // Load configuration
        if (!args.config) {
            console.error('Error: --config argument required');
            console.log('Usage: npm run search -- --config config.json');
            process.exit(1);
        }

        const config = loadConfig(args.config);

        // Initialize Solana client
        const solana = new SolanaClient(config);

        // Initialize consumer
        const consumer = new Consumer(config, solana);

        // Build search filters
        const filters = {};
        if (args.category) filters.category = args.category;
        if (args.tag) filters.tag = args.tag;
        if (args['max-price']) filters.maxPrice = parseFloat(args['max-price']);

        console.log('Searching for APIs...\n');

        // Search
        const results = await consumer.searchApis(filters);

        if (results.length === 0) {
            console.log('No APIs found.');
            return;
        }

        console.log(`Found ${results.length} API(s):\n`);

        for (let i = 0; i < results.length; i++) {
            const api = results[i];
            console.log(`${i + 1}. ${api.name} (${api.id})`);
            console.log(`   Provider: ${api.endpoint}`);
            console.log(`   Price: ${api.price} ${config.token.symbol} per ${api.unit}`);
            console.log(`   Queue: ${api.currentQueue || 0} waiting`);
            console.log(`   Description: ${api.description}`);
            if (api.category) {
                console.log(`   Category: ${api.category.join(', ')}`);
            }
            if (api.tags) {
                console.log(`   Tags: ${api.tags.join(', ')}`);
            }
            console.log();
        }

    } catch (error) {
        console.error('Error:', error.message);
        process.exit(1);
    }
}

main();
