#!/usr/bin/env node

/**
 * Consumer CLI - Request queue position
 *
 * Usage: npm run queue <api-id> -- --config config.json --provider http://provider-url
 */

const minimist = require('minimist');
const { loadConfig } = require('../utils/config');
const SolanaClient = require('../solana');
const Consumer = require('../consumer');

async function main() {
    const args = minimist(process.argv.slice(2));

    try {
        // Get API ID
        const apiId = args._[0];
        if (!apiId) {
            console.error('Error: API ID required');
            console.log('Usage: npm run queue <api-id> -- --config config.json --provider http://provider-url');
            process.exit(1);
        }

        // Load configuration
        if (!args.config) {
            console.error('Error: --config argument required');
            process.exit(1);
        }

        const config = loadConfig(args.config);

        // Get provider URL
        if (!args.provider) {
            console.error('Error: --provider argument required');
            console.log('Usage: npm run queue <api-id> -- --config config.json --provider http://provider-url');
            process.exit(1);
        }

        // Initialize consumer
        const solana = new SolanaClient(config);
        const consumer = new Consumer(config, solana);

        console.log(`Requesting queue position for ${apiId}...\n`);

        // Request queue
        const queueData = await consumer.requestQueue(args.provider, apiId);

        console.log('Queue Position Requested:');
        console.log(`Queue Code: ${queueData.code}`);
        console.log(`Position: ${queueData.position}`);
        console.log(`Price: ${queueData.price} ${config.token.symbol}`);
        console.log(`Expires in: ${queueData.expirySeconds} seconds`);
        console.log();
        console.log('Next steps:');
        console.log(`1. Send payment of ${queueData.price} ${config.token.symbol} to provider`);
        console.log(`2. Use the queue code and transaction signature to execute the API`);
        console.log();
        console.log(`Execute command:`);
        console.log(`npm run execute ${apiId} -- --config config.json --provider ${args.provider} --queue-code ${queueData.code} --body '{"key":"value"}'`);

    } catch (error) {
        console.error('Error:', error.message);
        process.exit(1);
    }
}

main();
