#!/usr/bin/env node

/**
 * Consumer CLI - Get job status
 *
 * Usage: npm run job <job-id> -- --config config.json --provider http://provider-url
 */

const minimist = require('minimist');
const { loadConfig } = require('../utils/config');
const SolanaClient = require('../solana');
const Consumer = require('../consumer');

async function main() {
    const args = minimist(process.argv.slice(2));

    try {
        // Get job ID
        const jobId = args._[0];
        if (!jobId) {
            console.error('Error: Job ID required');
            console.log('Usage: npm run job <job-id> -- --config config.json --provider http://provider-url');
            process.exit(1);
        }

        // Load configuration
        if (!args.config) {
            console.error('Error: --config argument required');
            process.exit(1);
        }

        const config = loadConfig(args.config);

        // Get provider URL
        if (!args.provider) {
            console.error('Error: --provider argument required');
            console.log('Usage: npm run job <job-id> -- --config config.json --provider http://provider-url');
            process.exit(1);
        }

        // Initialize consumer
        const solana = new SolanaClient(config);
        const consumer = new Consumer(config, solana);

        console.log(`Getting status for job ${jobId}...\n`);

        // Get job status
        const job = await consumer.getJobStatus(args.provider, jobId);

        console.log(`Job ID: ${job.jobId}`);
        console.log(`API: ${job.apiId}`);
        console.log(`Status: ${job.status}`);
        console.log(`Progress: ${job.progress}%`);
        console.log(`Message: ${job.progressMessage}`);
        console.log();

        if (job.createdAt) {
            console.log(`Created: ${new Date(job.createdAt).toISOString()}`);
        }
        if (job.startedAt) {
            console.log(`Started: ${new Date(job.startedAt).toISOString()}`);
        }
        if (job.completedAt) {
            console.log(`Completed: ${new Date(job.completedAt).toISOString()}`);
        }

        if (job.status === 'completed') {
            console.log('\nResult:');
            if (job.result) {
                console.log(JSON.stringify(job.result, null, 2));
            }
            if (job.resultUrl) {
                console.log(`\nResult URL: ${job.resultUrl}`);
                console.log(`Download: npm run download ${job.jobId} -- --config config.json --url ${job.resultUrl} --output result.file`);
            }
        }

        if (job.status === 'failed') {
            console.log(`\nError: ${job.error}`);
        }

    } catch (error) {
        console.error('Error:', error.message);
        process.exit(1);
    }
}

main();
