#!/usr/bin/env node

/**
 * Consumer CLI - Execute API (full workflow: queue -> pay -> execute -> poll)
 *
 * Usage:
 * Full workflow:
 *   npm run execute <api-id> -- --config config.json --provider http://provider-url --provider-wallet ABC123 --body '{"key":"value"}'
 *
 * With existing queue code:
 *   npm run execute <api-id> -- --config config.json --provider http://provider-url --queue-code XYZ --body '{"key":"value"}'
 */

const minimist = require('minimist');
const { loadConfig } = require('../utils/config');
const SolanaClient = require('../solana');
const HybridConsumer = require('../consumer/hybrid-consumer');

async function main() {
    const args = minimist(process.argv.slice(2));
    let config = null;
    let consumer = null;

    try {
        // Get API ID
        const apiId = args._[0];
        if (!apiId) {
            console.error('Error: API ID required');
            console.log('Usage: npm run execute <api-id> -- --config config.json --provider http://provider-url --provider-wallet ABC --body \'{"key":"value"}\'');
            process.exit(1);
        }

        // Load configuration
        if (!args.config) {
            console.error('Error: --config argument required');
            process.exit(1);
        }

        config = loadConfig(args.config);

        // Get provider URL
        if (!args.provider) {
            console.error('Error: --provider argument required');
            process.exit(1);
        }

        // Get parameters
        if (!args.body) {
            console.error('Error: --body argument required (JSON string)');
            process.exit(1);
        }

        let params;
        try {
            params = JSON.parse(args.body);
        } catch (error) {
            console.error('Error: Invalid JSON in --body argument');
            process.exit(1);
        }

        // Initialize consumer with P2P support
        const solana = new SolanaClient(config);
        consumer = new HybridConsumer(config, solana);

        // Start P2P node if enabled
        if (config.p2p?.enabled) {
            console.log('Starting P2P node...');
            await consumer.start();
        }

        // Check if using existing queue code or full workflow
        if (args['queue-code'] && args['transaction']) {
            // Use existing queue code and transaction
            console.log('Executing with existing queue code and transaction...\n');

            const job = await consumer.executeApi(
                args.provider,
                apiId,
                args['queue-code'],
                args['transaction'],
                params
            );

            console.log(`Job created: ${job.jobId}`);
            console.log(`Status: ${job.status}\n`);

            // Poll for completion
            console.log('Polling for completion...');
            const completedJob = await consumer.pollForCompletion(args.provider, job.jobId);

            console.log('\n=== Job Completed ===\n');
            console.log(`Job ID: ${completedJob.jobId}`);
            console.log(`Status: ${completedJob.status}`);
            console.log(`Progress: ${completedJob.progress}%`);
            if (completedJob.result) {
                console.log('\nResult:');
                console.log(JSON.stringify(completedJob.result, null, 2));
            }
            if (completedJob.localFilePath) {
                console.log(`\nResult file: ${completedJob.localFilePath}`);
            } else if (completedJob.resultFilename) {
                console.log(`\nResult available. Download from provider: ${args.provider}/job/${completedJob.jobId}/download`);
            } else if (completedJob.resultUrl) {
                console.log(`\nResult URL: ${completedJob.resultUrl}`);
                console.log(`Download: npm run download ${completedJob.jobId} -- --config config.json --url ${completedJob.resultUrl} --output result.file`);
            }

        } else {
            // Full workflow
            if (!args['provider-wallet']) {
                console.error('Error: --provider-wallet argument required for full workflow');
                process.exit(1);
            }

            console.log('Starting full API call workflow...\n');

            const completedJob = await consumer.callApi(
                args.provider,
                args['provider-wallet'],
                apiId,
                params
            );

            console.log('\n=== Job Completed ===\n');
            console.log(`Job ID: ${completedJob.jobId}`);
            console.log(`Status: ${completedJob.status}`);
            console.log(`Progress: ${completedJob.progress}%`);
            if (completedJob.result) {
                console.log('\nResult:');
                console.log(JSON.stringify(completedJob.result, null, 2));
            }
            if (completedJob.localFilePath) {
                console.log(`\nResult file: ${completedJob.localFilePath}`);
            } else if (completedJob.resultUrl) {
                console.log(`\nResult URL: ${completedJob.resultUrl}`);
                console.log(`Download: npm run download ${completedJob.jobId} -- --config config.json --url ${completedJob.resultUrl} --output result.file`);
            }
        }

    } catch (error) {
        console.error('Error:', error.message);
        process.exit(1);
    } finally {
        // Cleanup P2P node
        if (config?.p2p?.enabled && consumer) {
            try {
                await consumer.stop();
            } catch (e) {
                // Ignore cleanup errors
            }
        }
    }
}

main();
