#!/usr/bin/env node

/**
 * Consumer CLI - Download result file
 *
 * Usage: npm run download <job-id> -- --config config.json --url http://result-url --output result.file
 */

const minimist = require('minimist');
const { loadConfig } = require('../utils/config');
const SolanaClient = require('../solana');
const Consumer = require('../consumer');

async function main() {
    const args = minimist(process.argv.slice(2));

    try {
        // Get job ID (optional, for display purposes)
        const jobId = args._[0];

        // Load configuration
        if (!args.config) {
            console.error('Error: --config argument required');
            console.log('Usage: npm run download <job-id> -- --config config.json --url http://result-url --output result.file');
            process.exit(1);
        }

        const config = loadConfig(args.config);

        // Get result URL
        if (!args.url) {
            console.error('Error: --url argument required');
            console.log('Usage: npm run download <job-id> -- --config config.json --url http://result-url --output result.file');
            process.exit(1);
        }

        // Get output path
        if (!args.output) {
            console.error('Error: --output argument required');
            console.log('Usage: npm run download <job-id> -- --config config.json --url http://result-url --output result.file');
            process.exit(1);
        }

        // Initialize consumer
        const solana = new SolanaClient(config);
        const consumer = new Consumer(config, solana);

        console.log(`Downloading result${jobId ? ` for job ${jobId}` : ''}...`);
        console.log(`URL: ${args.url}`);
        console.log(`Output: ${args.output}\n`);

        // Download
        await consumer.downloadResult(args.url, args.output);

        console.log('✓ Download complete!');

    } catch (error) {
        console.error('Error:', error.message);
        process.exit(1);
    }
}

main();
