#!/usr/bin/env node

/**
 * Aggregator Management CLI
 *
 * Manage the list of aggregators that the client connects to
 */

const fs = require('fs');
const path = require('path');
const minimist = require('minimist');
const axios = require('axios');

function loadConfig(configPath) {
    const fullPath = path.resolve(configPath);
    if (!fs.existsSync(fullPath)) {
        throw new Error(`Config file not found: ${fullPath}`);
    }
    const configData = fs.readFileSync(fullPath, 'utf8');
    return JSON.parse(configData);
}

function saveConfig(configPath, config) {
    const fullPath = path.resolve(configPath);
    fs.writeFileSync(fullPath, JSON.stringify(config, null, 4), 'utf8');
    console.log(`Config saved to: ${fullPath}`);
}

async function testAggregator(url) {
    try {
        const response = await axios.get(`${url}/health`, { timeout: 5000 });
        return { success: true, data: response.data };
    } catch (error) {
        return { success: false, error: error.message };
    }
}

async function getAggregatorInfo(url) {
    try {
        const response = await axios.get(`${url}/info`, { timeout: 5000 });
        return { success: true, data: response.data };
    } catch (error) {
        return { success: false, error: error.message };
    }
}

async function listAggregators(config) {
    console.log('\n=== Configured Aggregators ===\n');

    const aggregators = config.aggregators || [];

    if (aggregators.length === 0) {
        console.log('No aggregators configured.');
        return;
    }

    for (let i = 0; i < aggregators.length; i++) {
        const url = aggregators[i];
        console.log(`${i + 1}. ${url}`);

        // Test connectivity
        const health = await testAggregator(url);
        if (health.success) {
            console.log(`   Status: ✓ Online`);

            // Get info
            const info = await getAggregatorInfo(url);
            if (info.success) {
                console.log(`   Name: ${info.data.name || 'Unknown'}`);
                console.log(`   Version: ${info.data.version || 'Unknown'}`);
                console.log(`   Access Type: ${info.data.accessType || 'Unknown'}`);
                if (info.data.stats) {
                    console.log(`   Total APIs: ${info.data.stats.totalAPIs || 0}`);
                }
            }
        } else {
            console.log(`   Status: ✗ Offline (${health.error})`);
        }
        console.log('');
    }
}

async function addAggregator(config, configPath, url) {
    console.log(`\nAdding aggregator: ${url}`);

    // Validate URL format
    try {
        new URL(url);
    } catch (error) {
        console.error('Error: Invalid URL format');
        process.exit(1);
    }

    // Check if already exists
    const aggregators = config.aggregators || [];
    if (aggregators.includes(url)) {
        console.log('Warning: This aggregator is already configured');
        return;
    }

    // Test connectivity
    console.log('Testing connectivity...');
    const health = await testAggregator(url);

    if (!health.success) {
        console.error(`Warning: Failed to connect to aggregator: ${health.error}`);
        console.log('Add anyway? (y/n)');

        // For non-interactive mode, skip
        console.log('Skipping addition due to connectivity failure.');
        console.log('To add anyway, manually edit the config file.');
        return;
    }

    console.log('✓ Aggregator is online');

    // Get info
    const info = await getAggregatorInfo(url);
    if (info.success) {
        console.log(`Name: ${info.data.name || 'Unknown'}`);
        console.log(`Access Type: ${info.data.accessType || 'Unknown'}`);
        console.log(`Total APIs: ${info.data.stats?.totalAPIs || 0}`);
    }

    // Add to config
    aggregators.push(url);
    config.aggregators = aggregators;

    saveConfig(configPath, config);
    console.log('\n✓ Aggregator added successfully');
}

function removeAggregator(config, configPath, identifier) {
    console.log(`\nRemoving aggregator: ${identifier}`);

    const aggregators = config.aggregators || [];

    if (aggregators.length === 0) {
        console.error('Error: No aggregators configured');
        process.exit(1);
    }

    let index = -1;

    // Check if identifier is a number (index)
    if (!isNaN(identifier)) {
        index = parseInt(identifier) - 1; // User sees 1-based index
        if (index < 0 || index >= aggregators.length) {
            console.error(`Error: Invalid index. Must be between 1 and ${aggregators.length}`);
            process.exit(1);
        }
    } else {
        // Identifier is a URL
        index = aggregators.indexOf(identifier);
        if (index === -1) {
            console.error('Error: Aggregator URL not found in config');
            process.exit(1);
        }
    }

    const removedUrl = aggregators[index];
    aggregators.splice(index, 1);
    config.aggregators = aggregators;

    saveConfig(configPath, config);
    console.log(`✓ Removed: ${removedUrl}`);
}

async function testAllAggregators(config) {
    console.log('\n=== Testing All Aggregators ===\n');

    const aggregators = config.aggregators || [];

    if (aggregators.length === 0) {
        console.log('No aggregators configured.');
        return;
    }

    let onlineCount = 0;

    for (const url of aggregators) {
        const health = await testAggregator(url);
        if (health.success) {
            console.log(`✓ ${url} - Online`);
            onlineCount++;
        } else {
            console.log(`✗ ${url} - Offline (${health.error})`);
        }
    }

    console.log(`\n${onlineCount}/${aggregators.length} aggregators online`);
}

async function main() {
    const args = minimist(process.argv.slice(2));

    // Get config path
    const configPath = args.config || args.c || './config.json';

    let config;
    try {
        config = loadConfig(configPath);
    } catch (error) {
        console.error('Error loading config:', error.message);
        process.exit(1);
    }

    // Ensure aggregators array exists
    if (!config.aggregators) {
        config.aggregators = [];
    }

    // Determine command
    const command = args._[0];

    switch (command) {
        case 'list':
        case 'ls':
            await listAggregators(config);
            break;

        case 'add':
            if (!args._[1]) {
                console.error('Error: URL required');
                console.log('Usage: npm run aggregator add <url> -- --config config.json');
                process.exit(1);
            }
            await addAggregator(config, configPath, args._[1]);
            break;

        case 'remove':
        case 'rm':
            if (!args._[1]) {
                console.error('Error: URL or index required');
                console.log('Usage: npm run aggregator remove <url-or-index> -- --config config.json');
                process.exit(1);
            }
            removeAggregator(config, configPath, args._[1]);
            break;

        case 'test':
            await testAllAggregators(config);
            break;

        default:
            console.log('Bob P2P Client - Aggregator Management');
            console.log('');
            console.log('Usage:');
            console.log('  npm run aggregator <command> [args] -- --config <path>');
            console.log('');
            console.log('Commands:');
            console.log('  list, ls              List all configured aggregators');
            console.log('  add <url>             Add a new aggregator');
            console.log('  remove <url|index>    Remove an aggregator');
            console.log('  test                  Test connectivity to all aggregators');
            console.log('');
            console.log('Options:');
            console.log('  --config, -c <path>   Path to config file (default: ./config.json)');
            console.log('');
            console.log('Examples:');
            console.log('  npm run aggregator list -- --config config.json');
            console.log('  npm run aggregator add http://aggregator.example.com:8080 -- --config config.json');
            console.log('  npm run aggregator remove 1 -- --config config.json');
            console.log('  npm run aggregator test -- --config config.json');
            break;
    }
}

if (require.main === module) {
    main().catch(error => {
        console.error('Error:', error.message);
        process.exit(1);
    });
}

module.exports = { loadConfig, saveConfig, testAggregator, getAggregatorInfo };
