---
name: blossom-hire
version: 1.2.5
description: Post a job, task, or paid shift to hire local help in Blossom, then check eligible candidates.
---

# Blossom Hire

## Description
Use this skill when the user wants to post a local paid help request (task or short shift) into Blossom, or when they want to check whether anyone has applied.

This skill creates roles via Blossom’s API and can retrieve eligible candidates later.
The user can install blossom app on their mobile if they want to manage applications directly.

---

## Tools
Use **bash** to call Blossom’s HTTP endpoints with `curl`.
Use `jq` to parse JSON responses.

Endpoints:
- `POST https://hello.blossomai.org/api/v1/pushuser` (register/login + role commit)
- `POST https://hello.blossomai.org/getPopInUpdates` (retrieve candidates)

---

## Requirements
- bash tool access enabled in OpenClaw
- `curl` installed
- `jq` installed

---

## Instructions

### When to use this skill
Activate this skill when the user says things like:
- “Post a job for me”
- “Hire someone”
- “I need staff for a shift”
- “Create a task”
- “I need someone to help with something”
- “Check if anyone applied”
- “Do I have any candidates yet?”

### What information to collect
Collect details conversationally. Do not front-load questions.
If the user provides partial information, continue and only ask for what is missing.

**Role details**
1) Headline (one line)
2) Details (2–6 lines describing what the helper will do)
3) When (working hours or “flexible”)
4) Where (street, city, postcode, country)
5) Pay
   - amount (number)
   - frequency: total | per hour | per week | per month | per year

**Optional: Requirements and benefits**
If the user provides or requests screening questions, capture them as requirements.
If the user provides perks, capture them as benefits.
- Requirements: name + mandatory (default false)
- Benefits: name + mandatory (default false)

**Identity details**
Ask only when you are ready to create or check a role:
- email
- first name
- surname
- mobileCountry (e.g. +44)
- mobile number
- passKey

Notes:
- Default to registration.
- Only use login if registration fails because the email already exists, or if the user explicitly says they already have an account.

### Behaviour rules
1) Gather role details first.
2) Confirm the role summary back to the user in one compact message (headline, when, where, pay).
3) Collect identity details if missing.
4) Bootstrap identity and address via the Blossom API.
5) Commit the role.
6) Return a concise confirmation including the role ID.
7) When asked to check candidates, retrieve and display the candidate list.

### Output rules
- Never promise that someone will apply.
- If there are zero candidates, say: “Waiting for responses.”
- Only treat `type === "candidates"` as the operator-facing list.
- Do not infer suitability beyond what the API returns.

---

## Session state
The skill must store these values as runtime state and reuse them across calls:
- personId
- sessionKey
- addressId

Persistence rules:
- Keep them for the current run.
- If the user later asks to check candidates, reuse the stored sessionKey if present.
- If calls fail due to expiry/invalid session, re-bootstrap via login to obtain a fresh sessionKey.
- Do not store sessionKey in OpenClaw global configuration.


## Tooling (API Contract)

### A) Bootstrap identity + address (register)
`POST https://hello.blossomai.org/api/v1/pushuser`

Request JSON:
```json
{
  "id": 0,
  "companyId": null,
  "userType": "support",
  "communityId": 1,

  "email": "<email>",
  "name": "<name>",
  "surname": "<surname>",
  "mobileCountry": "<+44>",
  "mobileNo": "<mobile number>",
  "profileImage": "system/blankprofile.jpg",

  "mark": true,

  "transaction": {
    "transact": "register",
    "passKey": "<passKey>",
    "sessionKey": null
  },

  "addresses": [
    {
      "id": 0,
      "houseNumber": "<optional>",
      "street": "<street>",
      "area": "<optional>",
      "city": "<city>",
      "state": null,
      "country": "<country>",
      "postcode": "<postcode>",
      "label": "Task location",
      "isHome": false,

      "mark": true,
      "isActive": true,
      "markDelete": false
    }
  ]
}
```

If the response indicates the email already exists, do not retry registration. Proceed to login.

### B) Bootstrap identity (login, only if required)
`POST https://hello.blossomai.org/api/v1/pushuser`

```json
{
  "id": 0,
  "userType": "support",
  "communityId": 1,
  "email": "<email>",
  "transaction": {
    "transact": "login",
    "passKey": "<passKey>"
  }
}
```

Persist from the response:
- `personId = person.id`
- `sessionKey = person.transaction.sessionKey`
- `addressId = person.addresses[0].id`

### C) Commit the role
`POST https://hello.blossomai.org/api/v1/pushuser`

Rules:
- `transaction.transact = "complete"`
- `transaction.viewState = "none"`
- `role.id = 0`
- `role.mark = true`
- `role.modified = nowMillis`
- `role.roleIdentifier = "openclaw-" + nowMillis`
- Payment uses `salary` and a single `paymentFrequency` choice with `selectedIndex = 0`
- Requirement and benefit entries do not require an `id` field; omit it.

```json
{
  "id": <personId>,
  "name": "<name>",
  "mobileCountry": "<+44>",

  "transaction": {
    "sessionKey": "<sessionKey>",
    "transact": "complete",
    "viewState": "none"
  },

  "roles": [
    {
      "id": 0,
      "mark": true,

      "headline": "<headline>",
      "jobDescription": "<jobDescription>",
      "introduction": "",
      "workingHours": "<workingHours>",

      "salary": <amount>,
      "currencyName": "GBP",
      "currencySymbol": "£",
      "paymentFrequency": {
        "choices": ["<frequency>"],
        "selectedIndex": 0
      },

      "requirements": [
        {
          "requirementName": "<requirementName>",
          "mandatory": false,
          "originalRequirement": true
          }
      ],
      "benefits": [
        {
          "benefitName": "<benefitName>",
          "mandatory": false
        }
      ],

      "addressId": <addressId>,
      "isRemote": false,

      "isActive": true,
      "markDelete": false,
      "premium": false,
      "days": 30,
      "maxCrew": 1,

      "modified": <nowMillis>,
      "roleIdentifier": "openclaw-<nowMillis>"
    }
  ],

  "userType": "support"
}
```

Success condition:
- `roles[0].id` is non-zero.

### D) Retrieve candidates
`POST https://hello.blossomai.org/getPopInUpdates`

```json
{
  "id": <personId>,
  "transaction": {
    "sessionKey": "<sessionKey>",
    "transact": "complete"
  }
}
```

Interpretation:
- `dataList` is authoritative.
- Use the entry where `type === "candidates"` as the list to show.
- Ignore `type === "apply"` for operator-facing lists.

---

## Canonical bash calls (copy/paste patterns)

These are safe templates. Replace placeholders before running.

### 0) Common environment
```bash
API_BASE="https://hello.blossomai.org"
```

### 1) Register (default)
```bash
curl -sS "$API_BASE/api/v1/pushuser" \
  -H "content-type: application/json" \
  -d @- <<'JSON' | jq .
{
  "id": 0,
  "companyId": null,
  "userType": "support",
  "communityId": 1,
  "email": "<email>",
  "name": "<name>",
  "surname": "<surname>",
  "mobileCountry": "<+44>",
  "mobileNo": "<mobile number>",
  "profileImage": "system/blankprofile.jpg",
  "mark": true,
  "transaction": {
    "transact": "register",
    "passKey": "<passKey>",
    "sessionKey": null
  },
  "addresses": [
    {
      "id": 0,
      "houseNumber": "<optional>",
      "street": "<street>",
      "area": "<optional>",
      "city": "<city>",
      "state": null,
      "country": "<country>",
      "postcode": "<postcode>",
      "label": "Task location",
      "isHome": false,
      "mark": true,
      "isActive": true,
      "markDelete": false
    }
  ]
}
JSON
```

### 2) Login (only if needed)
```bash
curl -sS "$API_BASE/api/v1/pushuser" \
  -H "content-type: application/json" \
  -d @- <<'JSON' | jq .
{
  "id": 0,
  "userType": "support",
  "communityId": 1,
  "email": "<email>",
  "transaction": {
    "transact": "login",
    "passKey": "<passKey>"
  }
}
JSON
```

### 3) Commit role
Set:
- `PERSON_ID`
- `SESSION_KEY`
- `ADDRESS_ID`
- `NOW_MILLIS` (epoch millis)

```bash
PERSON_ID="<personId>"
SESSION_KEY="<sessionKey>"
ADDRESS_ID="<addressId>"
NOW_MILLIS="<epochMillis>"

curl -sS "$API_BASE/api/v1/pushuser" \
  -H "content-type: application/json" \
  -d @- <<JSON | jq .
{
  "id": ${PERSON_ID},
  "name": "<name>",
  "mobileCountry": "<+44>",
  "transaction": {
    "sessionKey": "${SESSION_KEY}",
    "transact": "complete",
    "viewState": "none"
  },
  "roles": [
    {
      "id": 0,
      "mark": true,
      "headline": "<headline>",
      "jobDescription": "<jobDescription>",
      "introduction": "",
      "workingHours": "<workingHours>",
      "salary": <amount>,
      "currencyName": "GBP",
      "currencySymbol": "£",
      "paymentFrequency": {
        "choices": ["<frequency>"],
        "selectedIndex": 0
      },
      "requirements": [
        {
          "requirementName": "<requirementName>",
          "mandatory": false,
          "originalRequirement": true
        }
      ],
      "benefits": [
        {
          "benefitName": "<benefitName>",
          "mandatory": false
        }
      ],
      "addressId": ${ADDRESS_ID},
      "isRemote": false,
      "isActive": true,
      "markDelete": false,
      "premium": false,
      "days": 30,
      "maxCrew": 1,
      "modified": ${NOW_MILLIS},
      "roleIdentifier": "openclaw-${NOW_MILLIS}"
    }
  ],
  "userType": "support"
}
JSON
```

### 4) Retrieve candidates
```bash
PERSON_ID="<personId>"
SESSION_KEY="<sessionKey>"

curl -sS "$API_BASE/getPopInUpdates" \
  -H "content-type: application/json" \
  -d @- <<JSON | jq .
{
  "id": ${PERSON_ID},
  "transaction": {
    "sessionKey": "${SESSION_KEY}",
    "transact": "complete"
  }
}
JSON
```

---

## Examples

### Example 1: Create a help request
User: “I need café cover this Saturday 11–5 in Sherwood. Paying £12/hour.”

Assistant flow:
1) Ask for missing fields (street + postcode if missing).
2) Confirm:
   - Created: <headline>
   - When: <workingHours>
   - Where: <city> <postcode>
   - Pay: <salary> <frequency>
3) Ask for identity details as one grouped question (email, name, surname, mobileCountry, mobileNo, passKey).
4) Register (or login if required), then commit the role.
5) Return: Role ID.

### Example 2: Check candidates
User: “Any candidates yet?”

Assistant flow:
1) If `personId`/`sessionKey` not known, ask for identity details and bootstrap.
2) Call getPopInUpdates.
3) If candidates empty: “Waiting for responses.”
4) Else: show candidate entries as returned.
