---
name: better-ralph
description: "Run one Better Ralph iteration: PRD-driven autonomous coding. Read prd.json, pick next story, implement it, run checks, commit, mark story passed, append progress. Uses only standard OpenClaw tools (read, write, exec, git). Triggers on: run better ralph, better ralph iteration, do one ralph story, next prd story, ralph loop."
user-invocable: true
---

# Better Ralph – One Iteration (OpenClaw)

Execute **one iteration** of the Better Ralph workflow: pick the next PRD story, implement it, run quality checks, commit, update the PRD, and append progress. Uses only standard tools (read_file, write_file, edit, exec, git). No external runner or Aether-Claw required.

---

## When to Use

- User says: "run better ralph", "do one better ralph iteration", "next prd story", "ralph loop", "implement next story from prd".
- Project has a `prd.json` in the workspace root (see Output Format below for schema).

---

## One-Iteration Workflow

Do these steps in order. Use **only** your standard file, exec, and git tools.

### 1. Read state

- **Read** `prd.json` (workspace root). Parse the JSON.
- **Read** `progress.txt` if it exists. If it has a section `## Codebase Patterns` near the top (up to the next `##` or end of file), use that as context for implementation patterns. Otherwise proceed without it.

### 2. Pick the next story

- From `prd.json.userStories`, find all with `passes === false`.
- Sort by `priority` ascending (lower number = higher priority).
- Take the **first** (highest priority incomplete story).
- If **every** story has `passes === true`, reply: "All PRD stories are complete. Nothing left to do." and stop.

### 3. Ensure git branch

- Check current git branch (e.g. run `git branch --show-current` or use your git tool).
- If `prd.json` has a `branchName` and it differs from the current branch, checkout or create that branch (e.g. `git checkout -b <branchName>` or `git checkout <branchName>`).

### 4. Implement the story

- **Story** = the one you picked. It has: `id`, `title`, `description`, `acceptanceCriteria`, `priority`, `passes`.
- Implement the story: write or edit code so that every item in `acceptanceCriteria` is satisfied.
- Work on **this story only**. Do not start the next story.

### 5. Run quality checks

- Run the project’s quality commands (e.g. `npm test`, `npm run lint`, `npm run typecheck`, or whatever the project uses).
- If **any check fails**, do **not** commit. Tell the user what failed and stop. Do not update `prd.json` or `progress.txt` for a failed story.

### 6. Commit (only if checks passed)

- Stage all changes (e.g. `git add -A` or your git tool’s equivalent).
- Commit with message exactly: `feat: [Story ID] - [Story Title]`  
  Example: `feat: US-002 - Display priority on task cards`

### 7. Mark story passed in prd.json

- **Read** `prd.json` again (in case it changed).
- Find the user story with the same `id` you just completed. Set its `passes` to `true`.
- **Write** the full updated `prd.json` back (preserve structure and other fields; only change that story’s `passes`).

### 8. Append progress to progress.txt

- **Append** (do not overwrite) a new block to `progress.txt` with this format:

```
## [Current date/time] - [Story ID]
- What was implemented (1–2 sentences)
- Files changed (list paths)
- **Learnings for future iterations:**
  - Patterns or gotchas (e.g. "this codebase uses X for Y", "remember to update Z when changing W")
---
```

- If `progress.txt` does not exist, create it with a first line like `# Better Ralph Progress` then the block above.

### 9. Report to user

- Say which story you completed (ID and title) and that you updated the PRD and progress.
- If there are still stories with `passes === false`, say: "Run another iteration to do the next story." If all are complete, say: "All PRD stories are complete."

---

## prd.json format

If the user wants to **create** a new `prd.json` (no file yet), create it with this shape:

```json
{
  "project": "ProjectName",
  "branchName": "ralph/feature-kebab-case",
  "description": "Short feature description",
  "userStories": [
    {
      "id": "US-001",
      "title": "Short title",
      "description": "As a [role], I want [thing] so that [benefit].",
      "acceptanceCriteria": [
        "Verifiable criterion 1",
        "Verifiable criterion 2",
        "Typecheck passes"
      ],
      "priority": 1,
      "passes": false,
      "notes": ""
    }
  ]
}
```

- **priority**: Lower number = higher priority. Order by dependency (e.g. schema before UI).
- **passes**: Start as `false`; set to `true` only after the story is implemented and committed.
- **acceptanceCriteria**: Each item must be checkable (e.g. "Typecheck passes", "Tests pass").

---

## Codebase Patterns (progress.txt)

Optionally keep a **Codebase Patterns** section at the **top** of `progress.txt` so future iterations (or you in the next run) see it first:

```
# Better Ralph Progress

## Codebase Patterns
- Use X for Y in this codebase
- Always run Z after changing W
- Tests require PORT=3000

---
```

When you read `progress.txt` at the start of an iteration, use this section as context. When you discover a **reusable** pattern, add it here (edit the top of the file and keep the rest intact). Do not put story-specific details in Codebase Patterns.

---

## Rules

- **One story per invocation.** Do not implement multiple stories in one go.
- **Do not commit failing code.** Only commit after quality checks pass.
- **Do not mark a story as passed** if you did not commit (e.g. checks failed).
- **Append** to progress.txt; never replace the whole file (except when creating it for the first time).
- Keep changes **minimal and focused** on the current story’s acceptance criteria.

---

## Checklist (one iteration)

- [ ] Read prd.json and progress.txt (and Codebase Patterns if present)
- [ ] Picked next story (passes=false, lowest priority number)
- [ ] Git branch matches prd.json.branchName
- [ ] Implemented story and satisfied all acceptance criteria
- [ ] Quality checks passed (test/lint/typecheck)
- [ ] Committed with message `feat: [ID] - [Title]`
- [ ] Set that story’s passes to true in prd.json
- [ ] Appended progress block to progress.txt
