---
name: beehiiv
description: |
  beehiiv API integration with managed OAuth. Manage newsletter publications, subscriptions, posts, custom fields, segments, and automations.
  Use this skill when users want to manage newsletter subscribers, create posts, organize segments, or integrate with beehiiv publications.
  For other third party apps, use the api-gateway skill (https://clawhub.ai/byungkyu/api-gateway).
  Requires network access and valid Maton API key.
metadata:
  author: maton
  version: "1.0"
  clawdbot:
    emoji: 🧠
    requires:
      env:
        - MATON_API_KEY
---

# beehiiv

Access the beehiiv API with managed OAuth authentication. Manage newsletter publications, subscriptions, posts, custom fields, segments, tiers, and automations.

## Quick Start

```bash
# List publications
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/beehiiv/v2/publications')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

## Base URL

```
https://gateway.maton.ai/beehiiv/{native-api-path}
```

Replace `{native-api-path}` with the actual beehiiv API endpoint path. The gateway proxies requests to `api.beehiiv.com` and automatically injects your OAuth token.

## Authentication

All requests require the Maton API key in the Authorization header:

```
Authorization: Bearer $MATON_API_KEY
```

**Environment Variable:** Set your API key as `MATON_API_KEY`:

```bash
export MATON_API_KEY="YOUR_API_KEY"
```

### Getting Your API Key

1. Sign in or create an account at [maton.ai](https://maton.ai)
2. Go to [maton.ai/settings](https://maton.ai/settings)
3. Copy your API key

## Connection Management

Manage your beehiiv OAuth connections at `https://ctrl.maton.ai`.

### List Connections

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections?app=beehiiv&status=ACTIVE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Create Connection

```bash
python <<'EOF'
import urllib.request, os, json
data = json.dumps({'app': 'beehiiv'}).encode()
req = urllib.request.Request('https://ctrl.maton.ai/connections', data=data, method='POST')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Content-Type', 'application/json')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Get Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

**Response:**
```json
{
  "connection": {
    "connection_id": "8bfe17f4-0038-4cbd-afb4-907b1ffa9d66",
    "status": "ACTIVE",
    "creation_time": "2026-02-11T00:25:10.464852Z",
    "last_updated_time": "2026-02-11T00:27:00.816431Z",
    "url": "https://connect.maton.ai/?session_token=...",
    "app": "beehiiv",
    "metadata": {}
  }
}
```

Open the returned `url` in a browser to complete OAuth authorization.

### Delete Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}', method='DELETE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Specifying Connection

If you have multiple beehiiv connections, specify which one to use with the `Maton-Connection` header:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/beehiiv/v2/publications')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Maton-Connection', '8bfe17f4-0038-4cbd-afb4-907b1ffa9d66')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

If omitted, the gateway uses the default (oldest) active connection.

## API Reference

All beehiiv API endpoints follow this pattern:

```
/beehiiv/v2/{resource}
```

---

## Publications

### List Publications

```bash
GET /beehiiv/v2/publications
```

**Query Parameters:**

| Parameter | Description |
|-----------|-------------|
| `limit` | Results per page (1-100, default: 10) |
| `page` | Page number (default: 1) |
| `expand[]` | Expand with: `stats`, `stat_active_subscriptions`, `stat_average_open_rate`, etc. |
| `order_by` | Sort by: `created` or `name` |
| `direction` | Sort direction: `asc` or `desc` |

**Response:**
```json
{
  "data": [
    {
      "id": "pub_c6c521e4-91ac-4c14-8a52-06987b7e32f2",
      "name": "My Newsletter",
      "organization_name": "My Organization",
      "referral_program_enabled": true,
      "created": 1770767522
    }
  ],
  "page": 1,
  "limit": 10,
  "total_results": 1,
  "total_pages": 1
}
```

### Get Publication

```bash
GET /beehiiv/v2/publications/{publication_id}
```

---

## Subscriptions

### List Subscriptions

```bash
GET /beehiiv/v2/publications/{publication_id}/subscriptions
```

**Query Parameters:**

| Parameter | Description |
|-----------|-------------|
| `limit` | Results per page (1-100, default: 10) |
| `cursor` | Cursor for pagination (recommended) |
| `page` | Page number (deprecated, max 100 pages) |
| `email` | Filter by exact email (case-insensitive) |
| `status` | Filter: `validating`, `invalid`, `pending`, `active`, `inactive`, `all` |
| `tier` | Filter: `free`, `premium`, `all` |
| `expand[]` | Expand with: `stats`, `custom_fields`, `referrals` |
| `order_by` | Sort field (default: `created`) |
| `direction` | Sort direction: `asc` or `desc` |

**Response:**
```json
{
  "data": [
    {
      "id": "sub_c27d9640-f418-43a8-a0f9-528c20a05002",
      "email": "subscriber@example.com",
      "status": "active",
      "created": 1770767524,
      "subscription_tier": "free",
      "subscription_premium_tier_names": [],
      "utm_source": "direct",
      "utm_medium": "",
      "utm_channel": "website",
      "utm_campaign": "",
      "referring_site": "",
      "referral_code": "gBZbSVal1X",
      "stripe_customer_id": ""
    }
  ],
  "limit": 10,
  "has_more": false,
  "next_cursor": null
}
```

### Get Subscription by ID

```bash
GET /beehiiv/v2/publications/{publication_id}/subscriptions/{subscription_id}
```

**Query Parameters:**

| Parameter | Description |
|-----------|-------------|
| `expand[]` | Expand with: `stats`, `custom_fields`, `referrals`, `tags` |

### Get Subscription by Email

```bash
GET /beehiiv/v2/publications/{publication_id}/subscriptions/by_email/{email}
```

### Create Subscription

```bash
POST /beehiiv/v2/publications/{publication_id}/subscriptions
Content-Type: application/json

{
  "email": "newsubscriber@example.com",
  "utm_source": "api",
  "send_welcome_email": false,
  "reactivate_existing": false
}
```

**Request Body:**

| Field | Type | Required | Description |
|-------|------|----------|-------------|
| `email` | string | Yes | Subscriber email address |
| `reactivate_existing` | boolean | No | Reactivate if previously unsubscribed |
| `send_welcome_email` | boolean | No | Send welcome email |
| `utm_source` | string | No | UTM source for tracking |
| `utm_medium` | string | No | UTM medium |
| `referring_site` | string | No | Referral code of referring subscriber |
| `custom_fields` | object | No | Custom field values (fields must exist) |
| `double_opt_override` | string | No | `on` or `off` to override double opt-in |
| `tier` | string | No | Subscription tier |
| `premium_tier_names` | array | No | Premium tier names to assign |

### Update Subscription

```bash
PATCH /beehiiv/v2/publications/{publication_id}/subscriptions/{subscription_id}
Content-Type: application/json

{
  "utm_source": "updated-source",
  "custom_fields": [
    {"name": "First Name", "value": "John"}
  ]
}
```

### Delete Subscription

```bash
DELETE /beehiiv/v2/publications/{publication_id}/subscriptions/{subscription_id}
```

---

## Posts

### List Posts

```bash
GET /beehiiv/v2/publications/{publication_id}/posts
```

**Query Parameters:**

| Parameter | Description |
|-----------|-------------|
| `limit` | Results per page (1-100, default: 10) |
| `page` | Page number |
| `status` | Filter by status |
| `expand[]` | Expand with additional data |

**Response:**
```json
{
  "data": [],
  "page": 1,
  "limit": 10,
  "total_results": 0,
  "total_pages": 0
}
```

### Get Post

```bash
GET /beehiiv/v2/publications/{publication_id}/posts/{post_id}
```

### Delete Post

```bash
DELETE /beehiiv/v2/publications/{publication_id}/posts/{post_id}
```

---

## Custom Fields

### List Custom Fields

```bash
GET /beehiiv/v2/publications/{publication_id}/custom_fields
```

**Response:**
```json
{
  "data": [
    {
      "id": "95c9653f-a1cf-45f0-a140-97feef19057b",
      "kind": "string",
      "display": "Last Name",
      "created": 1770767523
    },
    {
      "id": "4cfe081e-c89b-4da5-9c1a-52a4fb8ba69e",
      "kind": "string",
      "display": "First Name",
      "created": 1770767523
    }
  ],
  "page": 1,
  "limit": 10,
  "total_results": 2,
  "total_pages": 1
}
```

**Field Kinds:** `string`, `integer`, `boolean`, `date`, `datetime`, `list`, `double`

### Create Custom Field

```bash
POST /beehiiv/v2/publications/{publication_id}/custom_fields
Content-Type: application/json

{
  "display": "Company",
  "kind": "string"
}
```

### Update Custom Field

```bash
PATCH /beehiiv/v2/publications/{publication_id}/custom_fields/{custom_field_id}
Content-Type: application/json

{
  "display": "Company Name"
}
```

### Delete Custom Field

```bash
DELETE /beehiiv/v2/publications/{publication_id}/custom_fields/{custom_field_id}
```

---

## Segments

### List Segments

```bash
GET /beehiiv/v2/publications/{publication_id}/segments
```

**Response:**
```json
{
  "data": [],
  "page": 1,
  "limit": 10,
  "total_results": 0,
  "total_pages": 0
}
```

### Get Segment

```bash
GET /beehiiv/v2/publications/{publication_id}/segments/{segment_id}
```

### Delete Segment

```bash
DELETE /beehiiv/v2/publications/{publication_id}/segments/{segment_id}
```

---

## Tiers

### List Tiers

```bash
GET /beehiiv/v2/publications/{publication_id}/tiers
```

### Get Tier

```bash
GET /beehiiv/v2/publications/{publication_id}/tiers/{tier_id}
```

### Create Tier

```bash
POST /beehiiv/v2/publications/{publication_id}/tiers
Content-Type: application/json

{
  "name": "Premium",
  "description": "Premium tier with exclusive content"
}
```

### Update Tier

```bash
PATCH /beehiiv/v2/publications/{publication_id}/tiers/{tier_id}
Content-Type: application/json

{
  "name": "Updated Tier Name"
}
```

---

## Automations

### List Automations

```bash
GET /beehiiv/v2/publications/{publication_id}/automations
```

### Get Automation

```bash
GET /beehiiv/v2/publications/{publication_id}/automations/{automation_id}
```

---

## Referral Program

### Get Referral Program

```bash
GET /beehiiv/v2/publications/{publication_id}/referral_program
```

---

## Pagination

beehiiv supports two pagination methods:

### Cursor-Based (Recommended)

```bash
GET /beehiiv/v2/publications/{publication_id}/subscriptions?limit=10&cursor={next_cursor}
```

**Response includes:**
```json
{
  "data": [...],
  "limit": 10,
  "has_more": true,
  "next_cursor": "eyJ0aW1lc3RhbXAiOiIyMDI0LTA3LTAyVDE3OjMwOjAwLjAwMDAwMFoifQ=="
}
```

Use the `next_cursor` value for subsequent requests.

### Page-Based (Deprecated)

```bash
GET /beehiiv/v2/publications?page=2&limit=10
```

**Response includes:**
```json
{
  "data": [...],
  "page": 2,
  "limit": 10,
  "total_results": 50,
  "total_pages": 5
}
```

**Note:** Page-based pagination is limited to 100 pages maximum.

## Code Examples

### JavaScript

```javascript
const response = await fetch(
  'https://gateway.maton.ai/beehiiv/v2/publications',
  {
    headers: {
      'Authorization': `Bearer ${process.env.MATON_API_KEY}`
    }
  }
);
const data = await response.json();
console.log(data.data);
```

### Python

```python
import os
import requests

response = requests.get(
    'https://gateway.maton.ai/beehiiv/v2/publications',
    headers={'Authorization': f'Bearer {os.environ["MATON_API_KEY"]}'}
)
data = response.json()
for pub in data['data']:
    print(f"{pub['id']}: {pub['name']}")
```

## Notes

- Publication IDs start with `pub_`
- Subscription IDs start with `sub_`
- Timestamps are Unix timestamps (seconds since epoch)
- Custom fields must be created before use in subscriptions
- Cursor-based pagination is recommended for better performance
- Page-based pagination is deprecated and limited to 100 pages
- IMPORTANT: When using curl commands, use `curl -g` when URLs contain brackets to disable glob parsing
- IMPORTANT: When piping curl output to `jq` or other commands, environment variables like `$MATON_API_KEY` may not expand correctly in some shell environments

## Error Handling

| Status | Meaning |
|--------|---------|
| 400 | Bad request or invalid parameters |
| 401 | Invalid or missing Maton API key |
| 403 | Forbidden - insufficient permissions or plan limitation |
| 404 | Resource not found |
| 429 | Rate limited |
| 500 | Internal server error |

### Troubleshooting: API Key Issues

1. Check that the `MATON_API_KEY` environment variable is set:

```bash
echo $MATON_API_KEY
```

2. Verify the API key is valid by listing connections:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Troubleshooting: Invalid App Name

1. Ensure your URL path starts with `beehiiv`. For example:

- Correct: `https://gateway.maton.ai/beehiiv/v2/publications`
- Incorrect: `https://gateway.maton.ai/v2/publications`

## Resources

- [beehiiv Developer Documentation](https://developers.beehiiv.com/)
- [beehiiv API Reference](https://developers.beehiiv.com/api-reference)
- [beehiiv Help Center](https://beehiivhelp.zendesk.com/)
- [Maton Community](https://discord.com/invite/dBfFAcefs2)
- [Maton Support](mailto:support@maton.ai)
