---
name: bear-blog-publisher
description: Publish blog posts to Bear Blog platform. Supports user-provided markdown, AI-generated content, and auto-generated diagrams.
---

# Bear Blog Publisher

Publish blog posts to Bear Blog (https://bearblog.dev/).

## Overview

This skill provides automated publishing capabilities for Bear Blog, including optional AI content generation and diagram generation.

## Authentication Methods (Choose One)

### Method 1: OpenClaw Config (Recommended for Personal Use)

Add to your `~/.openclaw/openclaw.json`:

```json
{
  "skills": {
    "bear-blog-publisher": {
      "email": "your@email.com",
      "password": "yourpassword"
    }
  }
}
```

**Security**: File permissions should be set to 600 (readable only by owner).

### Method 2: Environment Variables (Recommended for CI/CD)

```bash
export BEAR_BLOG_EMAIL="your@email.com"
export BEAR_BLOG_PASSWORD="yourpassword"
```

**Security**: Credentials exist only in memory, not written to disk.

### Method 3: Runtime Parameters (Recommended for Multi-User)

Provide credentials when calling the skill:

```python
publisher = BearBlogPublisher(email="user@example.com", password="secret")
```

**Security**: Caller (chat bot, web app, etc.) manages credential lifecycle.

## AI Content Generation (Optional)

To use AI content generation, configure one of the following:

### OpenAI

```bash
export OPENAI_API_KEY="sk-..."
```

### Kimi

```bash
export KIMI_API_KEY="your-kimi-api-key"
```

### Usage

```python
publisher = BearBlogPublisher()
content = publisher.generate_content(
    topic="Python best practices",
    provider="openai",  # or "kimi"
    tone="professional",
    length="medium"
)
result = publisher.publish(title="My Post", content=content)
```

## Priority Order

1. Runtime parameters (highest priority)
2. Environment variables
3. OpenClaw config (lowest priority)

## Capabilities

### 1. Publish Blog Post

**Input:**
- `title` (string): Blog post title
- `content` (string): Markdown content
- `email` (string, optional): Bear Blog email
- `password` (string, optional): Bear Blog password

**Output:**
- Published URL or error message

### 2. AI Content Generation (Optional)

Generate blog content using OpenAI or Kimi API.

### 3. Generate Diagram (Optional)

For technical topics, generates architecture diagrams using HTML/CSS + Playwright.

## Security Best Practices

1. **Never commit credentials to git**
2. **Use environment variables in production**
3. **Set file permissions to 600 for config files**
4. **Use runtime parameters for multi-user scenarios**

## Security Considerations

This skill makes several operational choices that users should be aware of:

### 1. Playwright Browser Download
- **Why**: Required for generating architecture diagrams as PNG images
- **Size**: ~100MB Chromium browser
- **Alternative**: Skip diagram generation if not needed

### 2. Temporary Files
- **Location**: `/tmp/diagram.html` and `/tmp/diagram.png`
- **Purpose**: Intermediate files for diagram generation
- **Cleanup**: Files are overwritten on each run, not explicitly deleted

### 3. `--no-sandbox` Flag
- **Why**: Required for running Chromium in containerized/Docker environments
- **Risk**: Slightly reduced browser isolation
- **Mitigation**: Only used for local HTML-to-image conversion, no external URLs loaded

### 4. Plaintext Password Storage (Optional)
- **Config file**: Only if user chooses Method 1
- **Recommendation**: Use environment variables (Method 2) or runtime parameters (Method 3) instead
- **If using config**: Always set file permissions to 600

## Example Usage

### With Config File

```bash
# ~/.openclaw/openclaw.json configured
You: "Publish a blog about Python tips"
AI: [Uses config credentials, publishes]
```

### With Environment Variables

```bash
export BEAR_BLOG_EMAIL="user@example.com"
export BEAR_BLOG_PASSWORD="secret"

You: "Publish a blog about Python tips"
AI: [Uses env vars, publishes]
```

### With AI Content Generation

```bash
export BEAR_BLOG_EMAIL="user@example.com"
export BEAR_BLOG_PASSWORD="secret"
export OPENAI_API_KEY="sk-..."

You: "Write and publish a blog about Python asyncio"
AI: [Generates content with OpenAI, publishes]
```

### With Runtime Parameters

```python
# In your chat bot code
email = get_user_email()  # Ask user
password = get_user_password()  # Ask user

publisher = BearBlogPublisher(email=email, password=password)
result = publisher.publish(title="My Post", content="# Content")
```

## Implementation

- Uses Bear Blog web API
- CSRF token authentication
- Session-based (no persistent storage)
- Playwright for diagram generation
- OpenAI/Kimi API for content generation

## License

MIT
