#!/usr/bin/env bash
set -euo pipefail

# Battle-Tested Agent — Workspace Audit
# Usage: audit.sh [workspace_dir]

WORKSPACE="${1:-$HOME/workspace}"
SCORE=0
TOTAL=12

echo "⚔️  Battle-Tested Agent — Workspace Audit"
echo "   Workspace: $WORKSPACE"
echo ""

check() {
  local name="$1"
  local status="$2"  # pass, fail, warn
  local detail="$3"

  case "$status" in
    pass)
      echo "  ✅ $(printf '%-25s' "$name") — $detail"
      SCORE=$((SCORE + 1))
      ;;
    fail)
      echo "  ❌ $(printf '%-25s' "$name") — $detail"
      ;;
    warn)
      echo "  ⚠️  $(printf '%-25s' "$name") — $detail"
      SCORE=$((SCORE + 1))  # partial credit
      ;;
  esac
}

# 1. WAL Protocol
if grep -q -i "WAL Protocol\|Write-Ahead Log\|write.*before.*respond\|write first.*respond second" "$WORKSPACE/SOUL.md" 2>/dev/null || \
   grep -q -i "WAL Protocol\|Write-Ahead Log" "$WORKSPACE/AGENTS.md" 2>/dev/null; then
  check "WAL Protocol" "pass" "Found in workspace files"
else
  check "WAL Protocol" "fail" "Not found in SOUL.md or AGENTS.md"
fi

# 2. Working Buffer
if [[ -f "$WORKSPACE/memory/working-buffer.md" ]]; then
  check "Working Buffer" "pass" "memory/working-buffer.md exists"
elif grep -q -i "working.buffer\|compaction.*survival\|danger.zone" "$WORKSPACE/SOUL.md" 2>/dev/null; then
  check "Working Buffer" "warn" "Referenced in SOUL.md but no buffer file"
else
  check "Working Buffer" "fail" "memory/working-buffer.md missing"
fi

# 3. Verification Rules
if grep -rq -i "no command.*no number\|include the command.*ran\|verification rule" "$WORKSPACE/memory/agents/" 2>/dev/null || \
   grep -q -i "verification rule\|no command.*no number" "$WORKSPACE/SOUL.md" 2>/dev/null; then
  check "Verification Rules" "pass" "Found in agent files"
else
  check "Verification Rules" "fail" "No verification rules in agent soul files"
fi

# 4. Verify Implementation
if grep -q -i "verify implementation.*not intent\|mechanism.*not.*text\|text changes.*behavior" "$WORKSPACE/SOUL.md" 2>/dev/null || \
   grep -q -i "verify implementation" "$WORKSPACE/QA.md" 2>/dev/null; then
  check "Verify Implementation" "pass" "Found in workspace files"
else
  check "Verify Implementation" "fail" "Not found in SOUL.md or QA.md"
fi

# 5. Decision Logs
if grep -q -i "decision.*reasoning\|decision.*log\|🧠 Decision" "$WORKSPACE/SOUL.md" 2>/dev/null || \
   grep -q -i "decision.*reasoning\|decision.*log" "$WORKSPACE/AGENTS.md" 2>/dev/null; then
  check "Decision Logs" "pass" "Found in workspace files"
else
  check "Decision Logs" "fail" "Not found in SOUL.md or AGENTS.md"
fi

# 6. Anti-Hallucination
if grep -q -i "anti.hallucination\|never invent.*task\|silence.*better.*hallucination\|OMIT IT" "$WORKSPACE/HEARTBEAT.md" 2>/dev/null || \
   grep -q -i "anti.hallucination" "$WORKSPACE/SOUL.md" 2>/dev/null; then
  check "Anti-Hallucination" "pass" "Found in heartbeat/soul files"
else
  check "Anti-Hallucination" "fail" "Not found in HEARTBEAT.md or SOUL.md"
fi

# 7. QA Gates
if [[ -f "$WORKSPACE/QA.md" ]]; then
  GATES=$(grep -c "Gate [0-9]" "$WORKSPACE/QA.md" 2>/dev/null || echo 0)
  if [[ "$GATES" -ge 3 ]]; then
    check "QA Gates" "pass" "QA.md with $GATES gates"
  else
    check "QA Gates" "warn" "QA.md exists but only $GATES gates (recommend 3+)"
  fi
else
  check "QA Gates" "fail" "QA.md missing"
fi

# 8. Self-Improvement
if [[ -d "$WORKSPACE/.learnings" ]]; then
  if grep -q -i "VFM\|Value-First Modification\|weighted score" "$WORKSPACE/.learnings/LEARNINGS.md" 2>/dev/null; then
    check "Self-Improvement" "pass" ".learnings/ with VFM scoring"
  else
    check "Self-Improvement" "warn" ".learnings/ exists but no VFM protocol"
  fi
else
  check "Self-Improvement" "fail" ".learnings/ directory missing"
fi

# 9. Delegation Rules
if grep -q -i "50.*lines\|spawn.*sub.agent\|delegation\|one task per agent" "$WORKSPACE/AGENTS.md" 2>/dev/null || \
   grep -q -i "delegation" "$WORKSPACE/SOUL.md" 2>/dev/null; then
  check "Delegation Rules" "pass" "Found in workspace files"
else
  check "Delegation Rules" "fail" "Not found in AGENTS.md or SOUL.md"
fi

# 10. Compaction Hardening
if grep -q -i "compaction.*injection\|compaction.*inert\|WORKFLOW_AUTO\|hallucinated injection" "$WORKSPACE/SOUL.md" 2>/dev/null; then
  check "Compaction Hardening" "pass" "Found in SOUL.md"
else
  check "Compaction Hardening" "fail" "Not found in SOUL.md"
fi

# 11. Simple Path First
if grep -q -i "simple path first\|dumbest.*version\|try before explain\|curl before MCP\|protocol.*framework.*plugin.*test\|Simple path.*before reaching" "$WORKSPACE/SOUL.md" 2>/dev/null; then
  check "Simple Path First" "pass" "Found in SOUL.md"
else
  check "Simple Path First" "fail" "Not found in SOUL.md"
fi

# 12. Unblock Before Shelve
if grep -q -i "unblock before shelve\|10 minutes.*blocker\|fix.*15 minutes.*shelving\|Unblock.*shelv\|spend.*minutes.*block" "$WORKSPACE/SOUL.md" 2>/dev/null || \
   grep -q -i "unblock before shelve\|spend.*minutes.*block" "$WORKSPACE/AGENTS.md" 2>/dev/null; then
  check "Unblock Before Shelve" "pass" "Found in workspace files"
else
  check "Unblock Before Shelve" "fail" "Not found in SOUL.md or AGENTS.md"
fi

echo ""
echo "  Score: $SCORE/$TOTAL patterns implemented"
echo ""

if [[ "$SCORE" -eq "$TOTAL" ]]; then
  echo "  🏆 Full coverage. Your agent is battle-tested."
elif [[ "$SCORE" -ge 9 ]]; then
  echo "  ⚔️  Strong. Add the missing patterns to reach full coverage."
elif [[ "$SCORE" -ge 6 ]]; then
  echo "  🔧 Good foundation. Prioritize the ❌ items — they're where failures happen."
else
  echo "  ⚠️  Early stage. Start with WAL Protocol, Verification Rules, and Anti-Hallucination."
fi
