import { describe, it, expect } from 'vitest';
import { listTodolistGroups, createTodolistGroup } from '../lib/api.js';

describe('Todolist Groups API', () => {
  describe('Function exports', () => {
    it('should export listTodolistGroups function', () => {
      expect(typeof listTodolistGroups).toBe('function');
    });

    it('should export createTodolistGroup function', () => {
      expect(typeof createTodolistGroup).toBe('function');
    });
  });

  describe('Function signatures', () => {
    it('listTodolistGroups should accept projectId and todolistId', () => {
      const fn = listTodolistGroups;
      expect(fn.length).toBeGreaterThanOrEqual(2);
    });

    it('createTodolistGroup should accept projectId, todolistId, name, and optional color', () => {
      const fn = createTodolistGroup;
      expect(fn.length).toBeGreaterThanOrEqual(3);
    });
  });

  describe('Type definitions', () => {
    it('should have proper async function signatures', () => {
      const listFn = listTodolistGroups.toString();
      expect(listFn).toContain('async');

      const createFn = createTodolistGroup.toString();
      expect(createFn).toContain('async');
    });
  });

  describe('API endpoint construction', () => {
    it('listTodolistGroups should construct correct endpoint pattern', () => {
      const fnStr = listTodolistGroups.toString();
      expect(fnStr).toContain('todolists');
      expect(fnStr).toContain('groups.json');
    });

    it('createTodolistGroup should construct correct endpoint pattern', () => {
      const fnStr = createTodolistGroup.toString();
      expect(fnStr).toContain('todolists');
      expect(fnStr).toContain('groups.json');
      expect(fnStr).toContain('post');
    });
  });

  describe('Parameter handling', () => {
    it('createTodolistGroup should handle optional color parameter', () => {
      const fnStr = createTodolistGroup.toString();
      expect(fnStr).toContain('color');
    });

    it('createTodolistGroup should build JSON payload with name', () => {
      const fnStr = createTodolistGroup.toString();
      expect(fnStr).toContain('name');
      expect(fnStr).toContain('json');
    });
  });

  describe('Error handling', () => {
    it('functions should use createClient for authentication', () => {
      const listFn = listTodolistGroups.toString();
      expect(listFn).toContain('createClient');

      const createFn = createTodolistGroup.toString();
      expect(createFn).toContain('createClient');
    });

    it('functions should use fetchAllPages for pagination', () => {
      const listFn = listTodolistGroups.toString();
      expect(listFn).toContain('fetchAllPages');
    });
  });

  describe('Response handling', () => {
    it('listTodolistGroups should use fetchAllPages for pagination', () => {
      const fnStr = listTodolistGroups.toString();
      expect(fnStr).toContain('fetchAllPages');
    });

    it('createTodolistGroup should parse JSON response', () => {
      const fnStr = createTodolistGroup.toString();
      expect(fnStr).toContain('.json()');
    });
  });

  describe('Integration with existing patterns', () => {
    it('should follow same pattern as other list functions', () => {
      const listFn = listTodolistGroups.toString();
      expect(listFn).toContain('fetchAllPages');
      expect(listFn).toContain('buckets');
    });

    it('should follow same pattern as other create functions', () => {
      const createFn = createTodolistGroup.toString();
      expect(createFn).toContain('post');
      expect(createFn).toContain('json');
    });
  });
});
