# Arbitrary Transaction Reference

Submit raw EVM transactions with explicit calldata to any supported chain.

## Supported Chains

| Chain | Chain ID |
|-------|----------|
| Ethereum | 1 |
| Polygon | 137 |
| Base | 8453 |
| Unichain | 130 |

## JSON Format

```json
{
  "to": "0x...",
  "data": "0x...",
  "value": "0",
  "chainId": 8453
}
```

| Field | Type | Required | Description |
|-------|------|----------|-------------|
| `to` | string | Yes | Target contract address (0x + 40 hex chars) |
| `data` | string | Yes | Calldata to execute (0x + hex string, or "0x" for empty) |
| `value` | string | Yes | Amount in wei (e.g., "0", "1000000000000000000" for 1 ETH) |
| `chainId` | number | Yes | Target chain ID (1, 137, 8453, or 130) |

## Validation Rules

| Field | Validation |
|-------|------------|
| `to` | Must be 0x followed by exactly 40 hex characters |
| `data` | Must start with 0x, can be "0x" for empty calldata |
| `value` | Wei amount as string, use "0" for no value transfer |
| `chainId` | Must be a supported chain ID |

## Prompt Examples

**Submit a raw transaction:**
```
Submit this transaction:
{
  "to": "0x1234567890abcdef1234567890abcdef12345678",
  "data": "0xa9059cbb000000000000000000000000recipient00000000000000000000000000000000000000000000000000000000000f4240",
  "value": "0",
  "chainId": 8453
}
```

**Execute calldata on a contract:**
```
Execute this calldata on Base:
{
  "to": "0xContractAddress...",
  "data": "0xFunctionSelector...",
  "value": "0",
  "chainId": 8453
}
```

**Send ETH with calldata:**
```
Submit transaction with value:
{
  "to": "0xRecipientAddress...",
  "data": "0x",
  "value": "1000000000000000000",
  "chainId": 1
}
```

**ERC-20 transfer via calldata:**
```
Submit this ERC-20 transfer:
{
  "to": "0xTokenContractAddress...",
  "data": "0xa9059cbb000000000000000000000000...",
  "value": "0",
  "chainId": 8453
}
```

## Common Issues

| Issue | Resolution |
|-------|------------|
| Unsupported chain | Use chainId 1, 137, 8453, or 130 |
| Invalid address | Ensure 0x + 40 hex chars |
| Invalid calldata | Ensure proper hex encoding with 0x prefix |
| Transaction reverted | Check calldata encoding and contract state |
| Insufficient funds | Ensure wallet has enough ETH/MATIC for gas + value |

## Use Cases

- **Custom contract interactions** - Call any function on any contract
- **Pre-built calldata execution** - Execute calldata generated by other tools
- **Advanced DeFi operations** - Complex multi-step transactions
- **Protocol integrations** - Interact with protocols not yet natively supported

## Best Practices

1. **Verify calldata** - Double-check encoding before submission
2. **Test on testnet first** - If possible, test transactions on testnets
3. **Start with zero value** - Test contract calls without sending ETH first
4. **Check gas estimates** - Ensure sufficient balance for gas costs
5. **Verify contract addresses** - Confirm target address is correct

## Security Notes

- **Irreversible** - Blockchain transactions cannot be undone
- **Verify everything** - Calldata determines exactly what happens
- **Trust the source** - Only execute calldata from trusted sources
- **Check value field** - Ensure you're not sending unintended ETH
- **Contract verification** - Confirm the target contract is legitimate
