"""Backboard SDK wrapper service."""
import asyncio
import os
from functools import wraps
from typing import Any, Callable, TypeVar

from backboard import BackboardClient
from backboard import (
    BackboardAPIError,
    BackboardNotFoundError,
    BackboardRateLimitError,
    BackboardValidationError,
)

T = TypeVar("T")


def run_async(func: Callable[..., T]) -> Callable[..., T]:
    """Decorator to run async functions synchronously."""

    @wraps(func)
    def wrapper(*args: Any, **kwargs: Any) -> T:
        return asyncio.run(func(*args, **kwargs))

    return wrapper


class BackboardService:
    """Service class for interacting with Backboard API."""

    def __init__(self) -> None:
        """Initialize the Backboard service."""
        api_key = os.environ.get("BACKBOARD_API_KEY")
        if not api_key:
            raise ValueError("BACKBOARD_API_KEY environment variable is not set")
        self._api_key = api_key

    def _get_client(self) -> BackboardClient:
        """Get a new Backboard client instance."""
        return BackboardClient(api_key=self._api_key)

    @staticmethod
    def _default_llm_provider() -> str:
        """Get the default LLM provider for Backboard."""
        return os.getenv("BACKBOARD_DEFAULT_LLM_PROVIDER", "openai")

    @staticmethod
    def _default_model_name() -> str:
        """Get the default model name for Backboard."""
        return os.getenv("BACKBOARD_DEFAULT_MODEL_NAME", "gpt-5.2")

    # === Assistant Operations ===

    @run_async
    async def create_assistant(
        self,
        name: str,
        system_prompt: str,
        tools: list[dict[str, Any]] | None = None,
        embedding_provider: str | None = None,
        embedding_model_name: str | None = None,
        embedding_dims: int | None = None,
    ) -> dict[str, Any]:
        """Create a new assistant."""
        client = self._get_client()
        kwargs: dict[str, Any] = {
            "name": name,
            "description": system_prompt,
        }
        if tools:
            kwargs["tools"] = tools
        if embedding_provider:
            kwargs["embedding_provider"] = embedding_provider
        if embedding_model_name:
            kwargs["embedding_model_name"] = embedding_model_name
        if embedding_dims:
            kwargs["embedding_dims"] = embedding_dims

        assistant = await client.create_assistant(**kwargs)
        return {
            "assistant_id": assistant.assistant_id,
            "name": assistant.name,
            "system_prompt": getattr(assistant, "description", None),
        }

    @run_async
    async def list_assistants(
        self, skip: int = 0, limit: int = 100
    ) -> list[dict[str, Any]]:
        """List all assistants."""
        client = self._get_client()
        assistants = await client.list_assistants(skip=skip, limit=limit)
        return [
            {
                "assistant_id": a.assistant_id,
                "name": a.name,
                "system_prompt": getattr(a, "description", None),
            }
            for a in assistants
        ]

    @run_async
    async def get_assistant(self, assistant_id: str) -> dict[str, Any]:
        """Get an assistant by ID."""
        client = self._get_client()
        assistant = await client.get_assistant(assistant_id)
        return {
            "assistant_id": assistant.assistant_id,
            "name": assistant.name,
            "system_prompt": getattr(assistant, "description", None),
        }

    @run_async
    async def update_assistant(
        self,
        assistant_id: str,
        name: str | None = None,
        system_prompt: str | None = None,
    ) -> dict[str, Any]:
        """Update an assistant."""
        client = self._get_client()
        kwargs: dict[str, Any] = {}
        if name:
            kwargs["name"] = name
        if system_prompt:
            kwargs["description"] = system_prompt

        assistant = await client.update_assistant(assistant_id, **kwargs)
        return {
            "assistant_id": assistant.assistant_id,
            "name": assistant.name,
            "system_prompt": getattr(assistant, "description", None),
        }

    @run_async
    async def delete_assistant(self, assistant_id: str) -> dict[str, Any]:
        """Delete an assistant."""
        client = self._get_client()
        result = await client.delete_assistant(assistant_id)
        return {"success": True, "message": "Assistant deleted successfully"}

    # === Thread Operations ===

    @run_async
    async def create_thread(self, assistant_id: str) -> dict[str, Any]:
        """Create a new thread for an assistant."""
        client = self._get_client()
        thread = await client.create_thread(assistant_id)
        return {
            "thread_id": thread.thread_id,
            "assistant_id": assistant_id,
        }

    @run_async
    async def list_threads(self, skip: int = 0, limit: int = 100) -> list[dict[str, Any]]:
        """List all threads."""
        client = self._get_client()
        threads = await client.list_threads(skip=skip, limit=limit)
        return [
            {
                "thread_id": t.thread_id,
                "assistant_id": getattr(t, "assistant_id", None),
            }
            for t in threads
        ]

    @run_async
    async def list_threads_for_assistant(
        self, assistant_id: str, skip: int = 0, limit: int = 100
    ) -> list[dict[str, Any]]:
        """List threads for a specific assistant."""
        client = self._get_client()
        threads = await client.list_threads_for_assistant(
            assistant_id, skip=skip, limit=limit
        )
        return [
            {
                "thread_id": t.thread_id,
                "assistant_id": assistant_id,
            }
            for t in threads
        ]

    @run_async
    async def get_thread(self, thread_id: str) -> dict[str, Any]:
        """Get a thread with its messages."""
        client = self._get_client()
        thread = await client.get_thread(thread_id)
        messages = []
        if hasattr(thread, "messages") and thread.messages:
            for m in thread.messages:
                messages.append(
                    {
                        "role": getattr(m, "role", "unknown"),
                        "content": getattr(m, "content", ""),
                    }
                )
        return {
            "thread_id": thread.thread_id,
            "assistant_id": getattr(thread, "assistant_id", None),
            "messages": messages,
        }

    @run_async
    async def delete_thread(self, thread_id: str) -> dict[str, Any]:
        """Delete a thread."""
        client = self._get_client()
        await client.delete_thread(thread_id)
        return {"success": True, "message": "Thread deleted successfully"}

    # === Message Operations ===

    @run_async
    async def add_message(
        self,
        thread_id: str,
        content: str,
        llm_provider: str | None = None,
        model_name: str | None = None,
        memory: str | None = "Auto",
    ) -> dict[str, Any]:
        """Send a message to a thread."""
        client = self._get_client()
        if not llm_provider:
            llm_provider = self._default_llm_provider()
        if not model_name:
            model_name = self._default_model_name()
        response = await client.add_message(
            thread_id=thread_id,
            content=content,
            llm_provider=llm_provider,
            model_name=model_name,
            memory=memory,
            stream=False,
        )
        result: dict[str, Any] = {
            "content": getattr(response, "content", None),
            "status": getattr(response, "status", None),
            "run_id": getattr(response, "run_id", None),
        }
        if hasattr(response, "tool_calls") and response.tool_calls:
            result["tool_calls"] = [
                {
                    "id": tc.id,
                    "function_name": tc.function.name,
                    "arguments": tc.function.parsed_arguments,
                }
                for tc in response.tool_calls
            ]
        return result

    # === Memory Operations ===

    @run_async
    async def add_memory(
        self,
        assistant_id: str,
        content: str,
        metadata: dict[str, Any] | None = None,
    ) -> dict[str, Any]:
        """Add a memory to an assistant."""
        client = self._get_client()
        kwargs: dict[str, Any] = {
            "assistant_id": assistant_id,
            "content": content,
        }
        if metadata:
            kwargs["metadata"] = metadata
        await client.add_memory(**kwargs)
        return {"success": True, "content": content}

    @run_async
    async def get_memories(self, assistant_id: str) -> list[dict[str, Any]]:
        """Get all memories for an assistant."""
        client = self._get_client()
        memories = await client.get_memories(assistant_id)
        return [
            {
                "id": m.id,
                "content": m.content,
                "metadata": getattr(m, "metadata", None),
            }
            for m in memories.memories
        ]

    @run_async
    async def get_memory(self, assistant_id: str, memory_id: str) -> dict[str, Any]:
        """Get a specific memory."""
        client = self._get_client()
        memory = await client.get_memory(assistant_id, memory_id)
        return {
            "id": memory.id,
            "content": memory.content,
            "metadata": getattr(memory, "metadata", None),
        }

    @run_async
    async def update_memory(
        self,
        assistant_id: str,
        memory_id: str,
        content: str,
    ) -> dict[str, Any]:
        """Update a memory."""
        client = self._get_client()
        await client.update_memory(
            assistant_id=assistant_id,
            memory_id=memory_id,
            content=content,
        )
        return {"success": True, "content": content}

    @run_async
    async def delete_memory(self, assistant_id: str, memory_id: str) -> dict[str, Any]:
        """Delete a memory."""
        client = self._get_client()
        await client.delete_memory(assistant_id, memory_id)
        return {"success": True, "message": "Memory deleted successfully"}

    @run_async
    async def get_memory_stats(self, assistant_id: str) -> dict[str, Any]:
        """Get memory statistics for an assistant."""
        client = self._get_client()
        stats = await client.get_memory_stats(assistant_id)
        return {"total_memories": stats.total_memories}

    # === Document Operations ===

    @run_async
    async def upload_document_to_assistant(
        self, assistant_id: str, file_path: str
    ) -> dict[str, Any]:
        """Upload a document to an assistant."""
        client = self._get_client()
        document = await client.upload_document_to_assistant(
            assistant_id=assistant_id,
            file_path=file_path,
        )
        return {
            "document_id": document.document_id,
            "filename": getattr(document, "filename", None),
            "status": getattr(document, "status", None),
        }

    @run_async
    async def upload_document_to_thread(
        self, thread_id: str, file_path: str
    ) -> dict[str, Any]:
        """Upload a document to a thread."""
        client = self._get_client()
        document = await client.upload_document_to_thread(
            thread_id=thread_id,
            file_path=file_path,
        )
        return {
            "document_id": document.document_id,
            "filename": getattr(document, "filename", None),
            "status": getattr(document, "status", None),
        }

    @run_async
    async def list_assistant_documents(
        self, assistant_id: str
    ) -> list[dict[str, Any]]:
        """List documents for an assistant."""
        client = self._get_client()
        documents = await client.list_assistant_documents(assistant_id)
        return [
            {
                "document_id": d.document_id,
                "filename": getattr(d, "filename", None),
                "status": getattr(d, "status", None),
            }
            for d in documents
        ]

    @run_async
    async def list_thread_documents(self, thread_id: str) -> list[dict[str, Any]]:
        """List documents for a thread."""
        client = self._get_client()
        documents = await client.list_thread_documents(thread_id)
        return [
            {
                "document_id": d.document_id,
                "filename": getattr(d, "filename", None),
                "status": getattr(d, "status", None),
            }
            for d in documents
        ]

    @run_async
    async def get_document_status(self, document_id: str) -> dict[str, Any]:
        """Get document processing status."""
        client = self._get_client()
        document = await client.get_document_status(document_id)
        return {
            "document_id": document.document_id,
            "filename": getattr(document, "filename", None),
            "status": getattr(document, "status", None),
        }

    @run_async
    async def delete_document(self, document_id: str) -> dict[str, Any]:
        """Delete a document."""
        client = self._get_client()
        await client.delete_document(document_id)
        return {"success": True, "message": "Document deleted successfully"}


# Export error classes for route handlers
__all__ = [
    "BackboardService",
    "BackboardAPIError",
    "BackboardNotFoundError",
    "BackboardRateLimitError",
    "BackboardValidationError",
]
