"""Thread management routes."""
from flask import Blueprint, jsonify, request

from backboard import (
    BackboardAPIError,
    BackboardNotFoundError,
    BackboardValidationError,
)

from ..models.schemas import MessageCreate
from ..services.backboard import BackboardService

bp = Blueprint("threads", __name__, url_prefix="/threads")


def get_service() -> BackboardService:
    """Get a BackboardService instance."""
    return BackboardService()


@bp.route("", methods=["GET"])
def list_threads():
    """List all threads."""
    try:
        skip = request.args.get("skip", 0, type=int)
        limit = request.args.get("limit", 100, type=int)
        service = get_service()
        result = service.list_threads(skip=skip, limit=limit)
        return jsonify({"threads": result})
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/<thread_id>", methods=["GET"])
def get_thread(thread_id: str):
    """Get a thread with its messages."""
    try:
        service = get_service()
        result = service.get_thread(thread_id)
        return jsonify(result)
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Thread not found"}), 404
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/<thread_id>", methods=["DELETE"])
def delete_thread(thread_id: str):
    """Delete a thread."""
    try:
        service = get_service()
        result = service.delete_thread(thread_id)
        return jsonify(result)
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Thread not found"}), 404
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/<thread_id>/messages", methods=["POST"])
def add_message(thread_id: str):
    """Send a message to a thread."""
    try:
        data = MessageCreate.model_validate(request.json)
        service = get_service()
        result = service.add_message(
            thread_id=thread_id,
            content=data.content,
            llm_provider=data.llm_provider,
            model_name=data.model_name,
            memory=data.memory,
        )
        return jsonify(result)
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Thread not found"}), 404
    except BackboardValidationError as e:
        return jsonify({"error": "Validation error", "detail": str(e)}), 400
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500
