"""Memory management routes."""
from flask import Blueprint, jsonify, request

from backboard import (
    BackboardAPIError,
    BackboardNotFoundError,
    BackboardValidationError,
)

from ..models.schemas import MemoryCreate, MemoryUpdate
from ..services.backboard import BackboardService

bp = Blueprint("memory", __name__, url_prefix="/assistants")


def get_service() -> BackboardService:
    """Get a BackboardService instance."""
    return BackboardService()


@bp.route("/<assistant_id>/memory", methods=["POST"])
def add_memory(assistant_id: str):
    """Add a memory to an assistant."""
    try:
        data = MemoryCreate.model_validate(request.json)
        service = get_service()
        result = service.add_memory(
            assistant_id=assistant_id,
            content=data.content,
            metadata=data.metadata,
        )
        return jsonify(result), 201
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Assistant not found"}), 404
    except BackboardValidationError as e:
        return jsonify({"error": "Validation error", "detail": str(e)}), 400
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/<assistant_id>/memory", methods=["GET"])
def get_memories(assistant_id: str):
    """Get all memories for an assistant."""
    try:
        service = get_service()
        result = service.get_memories(assistant_id)
        return jsonify({"memories": result})
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Assistant not found"}), 404
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/<assistant_id>/memory/<memory_id>", methods=["GET"])
def get_memory(assistant_id: str, memory_id: str):
    """Get a specific memory."""
    try:
        service = get_service()
        result = service.get_memory(assistant_id, memory_id)
        return jsonify(result)
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Memory not found"}), 404
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/<assistant_id>/memory/<memory_id>", methods=["PATCH"])
def update_memory(assistant_id: str, memory_id: str):
    """Update a memory."""
    try:
        data = MemoryUpdate.model_validate(request.json)
        service = get_service()
        result = service.update_memory(
            assistant_id=assistant_id,
            memory_id=memory_id,
            content=data.content,
        )
        return jsonify(result)
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Memory not found"}), 404
    except BackboardValidationError as e:
        return jsonify({"error": "Validation error", "detail": str(e)}), 400
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/<assistant_id>/memory/<memory_id>", methods=["DELETE"])
def delete_memory(assistant_id: str, memory_id: str):
    """Delete a memory."""
    try:
        service = get_service()
        result = service.delete_memory(assistant_id, memory_id)
        return jsonify(result)
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Memory not found"}), 404
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/<assistant_id>/memory/stats", methods=["GET"])
def get_memory_stats(assistant_id: str):
    """Get memory statistics for an assistant."""
    try:
        service = get_service()
        result = service.get_memory_stats(assistant_id)
        return jsonify(result)
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Assistant not found"}), 404
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500
