"""Document management routes."""
import os
import tempfile

from flask import Blueprint, jsonify, request

from backboard import (
    BackboardAPIError,
    BackboardNotFoundError,
    BackboardValidationError,
)

from ..services.backboard import BackboardService

bp = Blueprint("documents", __name__)


def get_service() -> BackboardService:
    """Get a BackboardService instance."""
    return BackboardService()


ALLOWED_EXTENSIONS = {
    "pdf",
    "docx",
    "xlsx",
    "pptx",
    "doc",
    "xls",
    "ppt",
    "txt",
    "csv",
    "md",
    "markdown",
    "py",
    "js",
    "html",
    "css",
    "xml",
    "json",
    "jsonl",
}


def allowed_file(filename: str) -> bool:
    """Check if file extension is allowed."""
    return "." in filename and filename.rsplit(".", 1)[1].lower() in ALLOWED_EXTENSIONS


@bp.route("/assistants/<assistant_id>/documents", methods=["POST"])
def upload_document_to_assistant(assistant_id: str):
    """Upload a document to an assistant."""
    try:
        if "file" not in request.files:
            return jsonify({"error": "No file provided"}), 400

        file = request.files["file"]
        if file.filename == "":
            return jsonify({"error": "No file selected"}), 400

        if not allowed_file(file.filename):
            return jsonify({"error": "File type not allowed"}), 400

        # Save to temp file and upload
        with tempfile.NamedTemporaryFile(
            delete=False, suffix=os.path.splitext(file.filename)[1]
        ) as tmp:
            file.save(tmp.name)
            service = get_service()
            result = service.upload_document_to_assistant(assistant_id, tmp.name)
            os.unlink(tmp.name)

        return jsonify(result), 201
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Assistant not found"}), 404
    except BackboardValidationError as e:
        return jsonify({"error": "Validation error", "detail": str(e)}), 400
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/assistants/<assistant_id>/documents", methods=["GET"])
def list_assistant_documents(assistant_id: str):
    """List documents for an assistant."""
    try:
        service = get_service()
        result = service.list_assistant_documents(assistant_id)
        return jsonify({"documents": result})
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Assistant not found"}), 404
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/threads/<thread_id>/documents", methods=["POST"])
def upload_document_to_thread(thread_id: str):
    """Upload a document to a thread."""
    try:
        if "file" not in request.files:
            return jsonify({"error": "No file provided"}), 400

        file = request.files["file"]
        if file.filename == "":
            return jsonify({"error": "No file selected"}), 400

        if not allowed_file(file.filename):
            return jsonify({"error": "File type not allowed"}), 400

        # Save to temp file and upload
        with tempfile.NamedTemporaryFile(
            delete=False, suffix=os.path.splitext(file.filename)[1]
        ) as tmp:
            file.save(tmp.name)
            service = get_service()
            result = service.upload_document_to_thread(thread_id, tmp.name)
            os.unlink(tmp.name)

        return jsonify(result), 201
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Thread not found"}), 404
    except BackboardValidationError as e:
        return jsonify({"error": "Validation error", "detail": str(e)}), 400
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/threads/<thread_id>/documents", methods=["GET"])
def list_thread_documents(thread_id: str):
    """List documents for a thread."""
    try:
        service = get_service()
        result = service.list_thread_documents(thread_id)
        return jsonify({"documents": result})
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Thread not found"}), 404
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/documents/<document_id>/status", methods=["GET"])
def get_document_status(document_id: str):
    """Get document processing status."""
    try:
        service = get_service()
        result = service.get_document_status(document_id)
        return jsonify(result)
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Document not found"}), 404
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/documents/<document_id>", methods=["DELETE"])
def delete_document(document_id: str):
    """Delete a document."""
    try:
        service = get_service()
        result = service.delete_document(document_id)
        return jsonify(result)
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Document not found"}), 404
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500
