"""Assistant management routes."""
from flask import Blueprint, jsonify, request

from backboard import (
    BackboardAPIError,
    BackboardNotFoundError,
    BackboardValidationError,
)

from ..models.schemas import AssistantCreate, AssistantUpdate
from ..services.backboard import BackboardService

bp = Blueprint("assistants", __name__, url_prefix="/assistants")


def get_service() -> BackboardService:
    """Get a BackboardService instance."""
    return BackboardService()


@bp.route("", methods=["POST"])
def create_assistant():
    """Create a new assistant."""
    try:
        data = AssistantCreate.model_validate(request.json)
        service = get_service()
        result = service.create_assistant(
            name=data.name,
            system_prompt=data.system_prompt,
            tools=[t.model_dump() for t in data.tools] if data.tools else None,
            embedding_provider=data.embedding_provider,
            embedding_model_name=data.embedding_model_name,
            embedding_dims=data.embedding_dims,
        )
        return jsonify(result), 201
    except BackboardValidationError as e:
        return jsonify({"error": "Validation error", "detail": str(e)}), 400
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("", methods=["GET"])
def list_assistants():
    """List all assistants."""
    try:
        skip = request.args.get("skip", 0, type=int)
        limit = request.args.get("limit", 100, type=int)
        service = get_service()
        result = service.list_assistants(skip=skip, limit=limit)
        return jsonify({"assistants": result})
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/<assistant_id>", methods=["GET"])
def get_assistant(assistant_id: str):
    """Get an assistant by ID."""
    try:
        service = get_service()
        result = service.get_assistant(assistant_id)
        return jsonify(result)
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Assistant not found"}), 404
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/<assistant_id>", methods=["PATCH"])
def update_assistant(assistant_id: str):
    """Update an assistant."""
    try:
        data = AssistantUpdate.model_validate(request.json)
        service = get_service()
        result = service.update_assistant(
            assistant_id=assistant_id,
            name=data.name,
            system_prompt=data.system_prompt,
        )
        return jsonify(result)
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Assistant not found"}), 404
    except BackboardValidationError as e:
        return jsonify({"error": "Validation error", "detail": str(e)}), 400
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/<assistant_id>", methods=["DELETE"])
def delete_assistant(assistant_id: str):
    """Delete an assistant."""
    try:
        service = get_service()
        result = service.delete_assistant(assistant_id)
        return jsonify(result)
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Assistant not found"}), 404
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/<assistant_id>/threads", methods=["POST"])
def create_thread_for_assistant(assistant_id: str):
    """Create a thread for an assistant."""
    try:
        service = get_service()
        result = service.create_thread(assistant_id)
        return jsonify(result), 201
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Assistant not found"}), 404
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500


@bp.route("/<assistant_id>/threads", methods=["GET"])
def list_threads_for_assistant(assistant_id: str):
    """List threads for an assistant."""
    try:
        skip = request.args.get("skip", 0, type=int)
        limit = request.args.get("limit", 100, type=int)
        service = get_service()
        result = service.list_threads_for_assistant(
            assistant_id, skip=skip, limit=limit
        )
        return jsonify({"threads": result})
    except BackboardNotFoundError:
        return jsonify({"error": "Not found", "detail": "Assistant not found"}), 404
    except BackboardAPIError as e:
        return jsonify({"error": "API error", "detail": str(e)}), 500
