"""Pydantic schemas for Backboard API entities."""
import os
from datetime import datetime
from typing import Any, Literal

from pydantic import BaseModel, Field


# === Assistant Schemas ===


class ToolDefinition(BaseModel):
    """Tool definition for an assistant."""

    type: Literal["function"] = "function"
    function: dict[str, Any]


class AssistantCreate(BaseModel):
    """Request schema for creating an assistant."""

    name: str = Field(..., min_length=1, max_length=255)
    system_prompt: str = Field(..., min_length=1)
    tools: list[ToolDefinition] | None = None
    embedding_provider: str | None = None
    embedding_model_name: str | None = None
    embedding_dims: int | None = None


class AssistantUpdate(BaseModel):
    """Request schema for updating an assistant."""

    name: str | None = Field(None, min_length=1, max_length=255)
    system_prompt: str | None = Field(None, min_length=1)


class AssistantResponse(BaseModel):
    """Response schema for an assistant."""

    assistant_id: str
    name: str
    system_prompt: str | None = None
    created_at: datetime | None = None


# === Thread Schemas ===


class ThreadCreate(BaseModel):
    """Request schema for creating a thread."""

    assistant_id: str


class MessageInThread(BaseModel):
    """Message within a thread response."""

    role: str
    content: str
    created_at: datetime | None = None


class ThreadResponse(BaseModel):
    """Response schema for a thread."""

    thread_id: str
    assistant_id: str
    created_at: datetime | None = None
    messages: list[MessageInThread] | None = None


# === Memory Schemas ===


class MemoryCreate(BaseModel):
    """Request schema for creating a memory."""

    content: str = Field(..., min_length=1)
    metadata: dict[str, Any] | None = None


class MemoryUpdate(BaseModel):
    """Request schema for updating a memory."""

    content: str = Field(..., min_length=1)


class MemoryResponse(BaseModel):
    """Response schema for a memory."""

    id: str
    content: str
    metadata: dict[str, Any] | None = None
    created_at: datetime | None = None


class MemoryListResponse(BaseModel):
    """Response schema for listing memories."""

    memories: list[MemoryResponse]


class MemoryStatsResponse(BaseModel):
    """Response schema for memory statistics."""

    total_memories: int


# === Document Schemas ===


class DocumentResponse(BaseModel):
    """Response schema for a document."""

    document_id: str
    filename: str | None = None
    status: str | None = None
    created_at: datetime | None = None


class DocumentListResponse(BaseModel):
    """Response schema for listing documents."""

    documents: list[DocumentResponse]


# === Message Schemas ===


class MessageCreate(BaseModel):
    """Request schema for sending a message."""

    content: str = Field(..., min_length=1)
    llm_provider: str | None = Field(
        default_factory=lambda: os.getenv("BACKBOARD_DEFAULT_LLM_PROVIDER", "openai")
    )
    model_name: str | None = Field(
        default_factory=lambda: os.getenv("BACKBOARD_DEFAULT_MODEL_NAME", "gpt-5.2")
    )
    memory: Literal["Auto", "Readonly", "off"] | None = "Auto"


class ToolCall(BaseModel):
    """Tool call from a message response."""

    id: str
    function_name: str
    arguments: dict[str, Any]


class MessageResponse(BaseModel):
    """Response schema for a message."""

    content: str | None = None
    status: str | None = None
    run_id: str | None = None
    tool_calls: list[ToolCall] | None = None


# === Generic Responses ===


class DeleteResponse(BaseModel):
    """Response schema for delete operations."""

    success: bool
    message: str


class ErrorResponse(BaseModel):
    """Response schema for errors."""

    error: str
    detail: str | None = None
