"""Flask application for Backboard OpenClaw Skill."""
import os

from dotenv import load_dotenv
from flask import Flask, jsonify

from pydantic import ValidationError

# Load environment variables
load_dotenv()


def create_app() -> Flask:
    """Create and configure the Flask application."""
    app = Flask(__name__)

    # Configuration
    app.config["JSON_SORT_KEYS"] = False

    # Register error handlers
    @app.errorhandler(ValidationError)
    def handle_validation_error(e: ValidationError):
        """Handle Pydantic validation errors."""
        return jsonify({"error": "Validation error", "detail": str(e)}), 400

    @app.errorhandler(ValueError)
    def handle_value_error(e: ValueError):
        """Handle value errors."""
        return jsonify({"error": "Invalid value", "detail": str(e)}), 400

    @app.errorhandler(404)
    def handle_not_found(e):
        """Handle 404 errors."""
        return jsonify({"error": "Not found"}), 404

    @app.errorhandler(500)
    def handle_internal_error(e):
        """Handle 500 errors."""
        return jsonify({"error": "Internal server error", "detail": str(e)}), 500

    # Health check endpoint
    @app.route("/health", methods=["GET"])
    def health():
        """Health check endpoint."""
        api_key_set = bool(os.environ.get("BACKBOARD_API_KEY"))
        return jsonify({
            "status": "healthy" if api_key_set else "unhealthy",
            "api_key_configured": api_key_set,
        })

    # Register blueprints
    from .routes import assistants, threads, memory, documents

    app.register_blueprint(assistants.bp)
    app.register_blueprint(threads.bp)
    app.register_blueprint(memory.bp)
    app.register_blueprint(documents.bp)

    return app


# Create app instance for Flask CLI
app = create_app()


if __name__ == "__main__":
    app.run(host="0.0.0.0", port=5100, debug=True)
