#!/bin/bash
# Phase 3: Objective Self-Scoring with PCTF compliance

set -euo pipefail

readonly SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")/.." && pwd)"
readonly WORKSPACE="${SCRIPT_DIR}/workspace"

echo "🐝 Phase 3: Objective Self-Scoring..."

# Scoring dimensions
declare -A MAX_SCORES=(
    [simplicity]=20
    [speed]=25
    [stability]=25
    [corner_cases]=20
    [maintainability]=10
)

for agent in a b c; do
    echo "  Agent ${agent^^} scoring..."
    
    agent_dir="${WORKSPACE}/run_${agent}"
    scorecard="${agent_dir}/SCORECARD.md"
    
    # PCTF-compliant scorecard
    cat > "$scorecard" << 'EOF'
# Scorecard - Agent [X]

## Self-Evaluation

### Detailed Scores

| Dimension | Max | Self-Score | Justification |
|-----------|-----|------------|---------------|
EOF

    # Add dimension rows
    for dim in simplicity speed stability corner_cases maintainability; do
        max="${MAX_SCORES[$dim]}"
        echo "| ${dim} | ${max} | X | [Why] |" >> "$scorecard"
    done
    
    cat >> "$scorecard" << 'EOF'
| **TOTAL** | **100** | **XX** | |

### Strengths
1. [Strength 1]
2. [Strength 2]

### Weaknesses
1. [Weakness 1]
2. [Weakness 2]

## Peer Evaluations

### Agent [Y] Evaluation of Me
- Score received: [X]/100
- Key criticisms: [summary]

### Agent [Z] Evaluation of Me
- Score received: [X]/100
- Key criticisms: [summary]

## My Evaluation of Peers

### Agent [Y]
| Dimension | Max | Score | Justification |
|-----------|-----|-------|---------------|
| simplicity | 20 | X | [Why] |
| speed | 25 | X | [Why] |
| stability | 25 | X | [Why] |
| corner_cases | 20 | X | [Why] |
| maintainability | 10 | X | [Why] |
| **TOTAL** | **100** | **XX** | |

### Agent [Z]
[Same format]

## Final Objective Conclusion

**Best Implementation:** [A/B/C/Mixed]

**Reasoning:**
[Data-driven justification based on scores]

**Recommendation for Final Delivery:**
[What should be delivered to user]

---
*Generated by b3ehive Phase 3*
EOF

    echo "    Created: SCORECARD.md"
done

# Linter: Verify all scorecards exist
echo "  Validating scorecards..."
for agent in a b c; do
    scorecard="${WORKSPACE}/run_${agent}/SCORECARD.md"
    
    # Assertion: Scorecard exists
    if [[ ! -f "$scorecard" ]]; then
        echo "ERROR: Agent ${agent} SCORECARD.md missing"
        exit 1
    fi
    
    # Assertion: Scorecard has required sections
    if ! grep -q "Self-Evaluation" "$scorecard"; then
        echo "ERROR: Agent ${agent} scorecard missing Self-Evaluation section"
        exit 1
    fi
    
    if ! grep -q "Final Objective Conclusion" "$scorecard"; then
        echo "ERROR: Agent ${agent} scorecard missing conclusion"
        exit 1
    fi
done

echo "✅ Phase 3 complete: 3 scorecards created and validated"
