#!/bin/bash
# Phase 2: Cross-Evaluation with PCTF compliance

set -euo pipefail

readonly SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")/.." && pwd)"
readonly WORKSPACE="${SCRIPT_DIR}/workspace"

echo "🐝 Phase 2: Cross-Evaluation..."

# Evaluation dimensions with weights
declare -A DIMENSIONS=(
    [simplicity]=20
    [speed]=25
    [stability]=25
    [corner_cases]=20
    [maintainability]=10
)

# Each agent evaluates the other two
for from in a b c; do
    for to in a b c; do
        [[ "$from" == "$to" ]] && continue
        
        echo "  Agent ${from^^} evaluating Agent ${to^^}..."
        
        eval_dir="${WORKSPACE}/run_${from}/evaluation"
        mkdir -p "$eval_dir"
        
        eval_file="${eval_dir}/EVALUATION_${from}_TO_${to}.md"
        
        # PCTF-compliant evaluation template
        cat > "$eval_file" << EOF
# Evaluation: Agent ${from^^} → Agent ${to^^}

## Executive Summary
[One paragraph summary]

## Metrics Comparison

| Dimension | Weight | Agent ${from^^} | Agent ${to^^} | Winner |
|-----------|--------|-----------------|---------------|--------|
EOF

        # Add dimension rows
        for dim in simplicity speed stability corner_cases maintainability; do
            weight="${DIMENSIONS[$dim]}"
            echo "| ${dim} | ${weight}% | TBD | TBD | TBD |" >> "$eval_file"
        done
        
        cat >> "$eval_file" << 'EOF'

## Detailed Analysis

### Code Simplicity
- Lines of code: [count]
- Cyclomatic complexity: [score]
- Readability: [1-10]

### Performance
- Time complexity: [Big-O]
- Space complexity: [Big-O]
- Benchmark results: [if available]

### Stability
- Error handling coverage: [%]
- Resource cleanup: [check]
- Fault tolerance: [test results]

### Corner Case Coverage
- Input validation: [comprehensive/partial/none]
- Boundary conditions: [covered/missed]
- Edge cases: [tested/missed]

## Issues Identified

### Issue 1: [Title]
- **Location:** [file:line]
- **Problem:** [description]
- **Impact:** [severity]
- **Fix:** [recommendation]

## Persuasion Argument

[Why Agent ${from^^}'s approach is objectively superior]

## Overall Score

Agent ${to^^}: **XX/100**

---
*Generated by b3ehive Phase 2*
EOF

        echo "    Created: EVALUATION_${from}_TO_${to}.md"
    done
done

# Linter: Verify all 6 evaluations exist
echo "  Validating evaluations..."
eval_count=0
for from in a b c; do
    for to in a b c; do
        [[ "$from" == "$to" ]] && continue
        
        eval_file="${WORKSPACE}/run_${from}/evaluation/EVALUATION_${from}_TO_${to}.md"
        if [[ -f "$eval_file" ]]; then
            ((eval_count++))
        else
            echo "ERROR: Evaluation ${from}→${to} missing"
            exit 1
        fi
    done
done

# Assertion: All 6 evaluations created
if [[ $eval_count -ne 6 ]]; then
    echo "ERROR: Expected 6 evaluations, found $eval_count"
    exit 1
fi

echo "✅ Phase 2 complete: 6 evaluations created and validated"
