#!/usr/bin/env bash
# =============================================================================
# ECS Production Health Monitor
#
# Generic health monitor for AWS ECS services. Checks:
#   1. Domain HTTP probe (if ECS_DOMAIN is set)
#   2. Custom HTTP endpoint probes (if ECS_HTTP_ENDPOINTS is set)
#   3. ECS service status (desired vs running task counts)
#   4. ALB target group health
#   5. SSL certificate expiry (if ECS_DOMAIN is set)
#
# Configuration (environment variables):
#   ECS_CLUSTER        (required) ECS cluster name
#   ECS_REGION         AWS region (default: us-east-1)
#   ECS_DOMAIN         Domain for HTTP/SSL checks (optional)
#   ECS_SERVICES       Comma-separated service names (auto-detect if unset)
#   ECS_HEALTH_STATE   State file path (default: ./data/ecs-health.json)
#   ECS_HEALTH_OUTDIR  Output directory (default: ./data/)
#   ECS_HTTP_ENDPOINTS Comma-separated name=url pairs for HTTP probes (optional)
#
# Usage:
#   ./scripts/ecs-health.sh              # Full check
#   ./scripts/ecs-health.sh --quiet      # No alerts, just status file
#   ./scripts/ecs-health.sh --json       # JSON output to stdout
#
# Exit codes:
#   0 = all healthy
#   1 = one or more checks failed
#   2 = script error (missing deps, missing config)
# =============================================================================
set -uo pipefail

# ---------------------------------------------------------------------------
# Configuration
# ---------------------------------------------------------------------------
if [[ -z "${ECS_CLUSTER:-}" ]]; then
  echo "ERROR: ECS_CLUSTER environment variable is required" >&2
  exit 2
fi

CLUSTER="${ECS_CLUSTER}"
REGION="${ECS_REGION:-us-east-1}"
DOMAIN="${ECS_DOMAIN:-}"
OUTDIR="${ECS_HEALTH_OUTDIR:-./data}"
STATUS_FILE="${ECS_HEALTH_STATE:-${OUTDIR}/ecs-health.json}"

QUIET=false
JSON_ONLY=false
for arg in "$@"; do
  case "$arg" in
    --quiet) QUIET=true ;;
    --json)  JSON_ONLY=true ;;
  esac
done

# Ensure output directory exists
mkdir -p "$(dirname "$STATUS_FILE")"
mkdir -p "$OUTDIR"

# Dependencies check
for cmd in curl aws python3; do
  if ! command -v "$cmd" &>/dev/null; then
    echo "Missing dependency: $cmd" >&2
    exit 2
  fi
done

# ---------------------------------------------------------------------------
# Auto-detect services from cluster
# ---------------------------------------------------------------------------
detect_services() {
  local svc_arns
  svc_arns=$(aws ecs list-services --cluster "$CLUSTER" --region "$REGION" --output json 2>/dev/null)
  if [[ -z "$svc_arns" ]]; then
    echo "" 
    return
  fi
  echo "$svc_arns" | python3 -c "
import json, sys
data = json.load(sys.stdin)
arns = data.get('serviceArns', [])
names = [a.split('/')[-1] for a in arns]
print(' '.join(names))
" 2>/dev/null
}

if [[ -n "${ECS_SERVICES:-}" ]]; then
  # Convert comma-separated to array
  IFS=',' read -ra SERVICES <<< "$ECS_SERVICES"
else
  read -ra SERVICES <<< "$(detect_services)"
fi

if [[ ${#SERVICES[@]} -eq 0 ]]; then
  echo "ERROR: No services found in cluster '$CLUSTER'. Set ECS_SERVICES or check cluster name." >&2
  exit 2
fi

# Timestamp
NOW=$(date +%s)
NOW_ISO=$(date -u +"%Y-%m-%dT%H:%M:%SZ")

# Collect results into temp directory
TMPDIR=$(mktemp -d)
trap 'rm -rf "$TMPDIR"' EXIT
OVERALL="healthy"

# ---------------------------------------------------------------------------
# 1. Domain HTTP Check (if ECS_DOMAIN is set)
# ---------------------------------------------------------------------------
if [[ -n "$DOMAIN" ]]; then
  fe_status=$(curl -s -o /dev/null -w "%{http_code}" --max-time 10 "https://$DOMAIN/" 2>/dev/null || echo "000")
  fe_latency=$(curl -s -o /dev/null -w "%{time_total}" --max-time 10 "https://$DOMAIN/" 2>/dev/null || echo "0")

  if [[ "$fe_status" == "200" ]]; then
    echo '{"status":"healthy","http_code":'"$fe_status"',"latency_s":'"$fe_latency"'}' > "$TMPDIR/domain_http.json"
  else
    echo '{"status":"unhealthy","http_code":'"$fe_status"',"latency_s":'"$fe_latency"',"error":"Expected 200, got '"$fe_status"'"}' > "$TMPDIR/domain_http.json"
    OVERALL="unhealthy"
  fi
fi

# ---------------------------------------------------------------------------
# 2. Custom HTTP Endpoint Probes (if ECS_HTTP_ENDPOINTS is set)
# ---------------------------------------------------------------------------
if [[ -n "${ECS_HTTP_ENDPOINTS:-}" ]]; then
  IFS=',' read -ra ENDPOINTS <<< "$ECS_HTTP_ENDPOINTS"
  for ep in "${ENDPOINTS[@]}"; do
    ep_name="${ep%%=*}"
    ep_url="${ep#*=}"
    ep_status=$(curl -s -o /dev/null -w "%{http_code}" --max-time 10 "$ep_url" 2>/dev/null || echo "000")
    ep_latency=$(curl -s -o /dev/null -w "%{time_total}" --max-time 10 "$ep_url" 2>/dev/null || echo "0")

    # Any non-zero HTTP response means reachable
    if [[ "$ep_status" != "000" && "$ep_status" != "" ]]; then
      echo '{"status":"healthy","http_code":'"$ep_status"',"latency_s":'"$ep_latency"',"url":"'"$ep_url"'"}' > "$TMPDIR/http_${ep_name}.json"
    else
      echo '{"status":"unhealthy","http_code":'"$ep_status"',"latency_s":'"$ep_latency"',"url":"'"$ep_url"'","error":"Unreachable"}' > "$TMPDIR/http_${ep_name}.json"
      OVERALL="unhealthy"
    fi
  done
fi

# ---------------------------------------------------------------------------
# 3. ECS Service Status
# ---------------------------------------------------------------------------
ecs_json=$(aws ecs describe-services \
  --cluster "$CLUSTER" \
  --services "${SERVICES[@]}" \
  --region "$REGION" \
  --output json 2>/dev/null)

if [[ -z "$ecs_json" ]]; then
  echo '{"status":"error","error":"Failed to query ECS"}' > "$TMPDIR/ecs_services.json"
  OVERALL="unhealthy"
else
  ecs_result=$(echo "$ecs_json" | python3 -c "
import json, sys

data = json.load(sys.stdin)
services = data.get('services', [])
failures = data.get('failures', [])

result = {'status': 'healthy', 'services': {}}
issues = []

for svc in services:
    name = svc['serviceName']
    desired = svc['desiredCount']
    running = svc['runningCount']
    status = svc['status']

    svc_health = 'healthy'

    if desired == 0:
        # Scaled down — not an error, just informational
        if status != 'ACTIVE':
            svc_health = 'unhealthy'
            issues.append(f'{name}: status={status}')
        else:
            svc_health = 'scaled_down'
    elif running < desired:
        svc_health = 'degraded'
        issues.append(f'{name}: {running}/{desired} tasks running')
    elif status != 'ACTIVE':
        svc_health = 'unhealthy'
        issues.append(f'{name}: status={status}')

    result['services'][name] = {
        'desired': desired,
        'running': running,
        'status': status,
        'health': svc_health
    }

for f in failures:
    arn = f.get('arn', 'unknown')
    reason = f.get('reason', 'unknown')
    name = arn.split('/')[-1] if '/' in arn else arn
    issues.append(f'{name}: FAILURE - {reason}')
    result['services'][name] = {'health': 'failed', 'error': reason}

if issues:
    result['status'] = 'degraded'
    result['issues'] = issues

print(json.dumps(result))
" 2>/dev/null)

  if [[ -n "$ecs_result" ]]; then
    echo "$ecs_result" > "$TMPDIR/ecs_services.json"
  else
    echo '{"status":"error","error":"Failed to parse ECS response"}' > "$TMPDIR/ecs_services.json"
  fi

  ecs_status=$(echo "$ecs_result" | python3 -c "import json,sys; print(json.load(sys.stdin).get('status','unknown'))" 2>/dev/null)
  if [[ "$ecs_status" == "degraded" || "$ecs_status" == "unhealthy" ]]; then
    [[ "$OVERALL" != "unhealthy" ]] && OVERALL="degraded"
  fi
fi

# ---------------------------------------------------------------------------
# 4. ALB Target Group Health
# ---------------------------------------------------------------------------
alb_result=$(python3 -c "
import json, subprocess, sys

tg_result = subprocess.run(
    ['aws', 'elbv2', 'describe-target-groups', '--region', '$REGION', '--output', 'json'],
    capture_output=True, text=True
)
if tg_result.returncode != 0:
    print(json.dumps({'status': 'error', 'error': 'Failed to query target groups'}))
    sys.exit(0)

tgs = json.loads(tg_result.stdout).get('TargetGroups', [])
result = {'status': 'healthy', 'target_groups': {}}
issues = []

for tg in tgs:
    name = tg['TargetGroupName']
    arn = tg['TargetGroupArn']

    health_result = subprocess.run(
        ['aws', 'elbv2', 'describe-target-health', '--target-group-arn', arn,
         '--region', '$REGION', '--output', 'json'],
        capture_output=True, text=True
    )
    if health_result.returncode != 0:
        result['target_groups'][name] = {'status': 'error', 'error': 'Query failed'}
        issues.append(f'{name}: failed to query health')
        continue

    targets = json.loads(health_result.stdout).get('TargetHealthDescriptions', [])
    healthy_count = sum(1 for t in targets if t['TargetHealth']['State'] == 'healthy')
    unhealthy = [t for t in targets if t['TargetHealth']['State'] not in ('healthy', 'draining', 'initial')]
    draining = sum(1 for t in targets if t['TargetHealth']['State'] == 'draining')
    total = len(targets)

    tg_status = 'healthy' if healthy_count > 0 else ('empty' if total == 0 else 'unhealthy')

    result['target_groups'][name] = {
        'healthy': healthy_count,
        'draining': draining,
        'unhealthy': len(unhealthy),
        'total': total,
        'status': tg_status
    }

    if unhealthy:
        for u in unhealthy:
            target_id = u['Target']['Id']
            state = u['TargetHealth']['State']
            reason = u['TargetHealth'].get('Reason', 'unknown')
            issues.append(f'{name}: {target_id} is {state} ({reason})')

    if total > 0 and healthy_count == 0:
        issues.append(f'{name}: no healthy targets ({total} total)')

if issues:
    result['status'] = 'degraded'
    result['issues'] = issues

print(json.dumps(result))
" 2>/dev/null)

if [[ -n "$alb_result" ]]; then
  echo "$alb_result" > "$TMPDIR/alb_targets.json"
else
  echo '{"status":"error","error":"Failed to check ALB targets"}' > "$TMPDIR/alb_targets.json"
fi

alb_status=$(echo "$alb_result" | python3 -c "import json,sys; print(json.load(sys.stdin).get('status','unknown'))" 2>/dev/null)
if [[ "$alb_status" == "unhealthy" ]]; then
  OVERALL="unhealthy"
elif [[ "$alb_status" == "degraded" && "$OVERALL" != "unhealthy" ]]; then
  OVERALL="degraded"
fi

# ---------------------------------------------------------------------------
# 5. SSL Certificate Check (if domain is set)
# ---------------------------------------------------------------------------
if [[ -n "$DOMAIN" ]] && command -v openssl &>/dev/null; then
  ssl_expiry=$(echo | openssl s_client -servername "$DOMAIN" -connect "$DOMAIN:443" 2>/dev/null | openssl x509 -noout -enddate 2>/dev/null | cut -d= -f2)
  if [[ -n "$ssl_expiry" ]]; then
    ssl_epoch=$(date -d "$ssl_expiry" +%s 2>/dev/null || echo "0")
    ssl_days_left=$(( (ssl_epoch - NOW) / 86400 ))

    if [[ $ssl_days_left -lt 7 ]]; then
      echo '{"status":"critical","expires":"'"$ssl_expiry"'","days_left":'"$ssl_days_left"'}' > "$TMPDIR/ssl.json"
      OVERALL="unhealthy"
    elif [[ $ssl_days_left -lt 30 ]]; then
      echo '{"status":"warning","expires":"'"$ssl_expiry"'","days_left":'"$ssl_days_left"'}' > "$TMPDIR/ssl.json"
    else
      echo '{"status":"healthy","expires":"'"$ssl_expiry"'","days_left":'"$ssl_days_left"'}' > "$TMPDIR/ssl.json"
    fi
  else
    echo '{"status":"error","error":"Could not retrieve SSL certificate"}' > "$TMPDIR/ssl.json"
  fi
fi

# ---------------------------------------------------------------------------
# Build Final JSON
# ---------------------------------------------------------------------------
FINAL_JSON=$(python3 -c "
import json, os, glob

tmpdir = '$TMPDIR'
checks = {}
for f in glob.glob(os.path.join(tmpdir, '*.json')):
    name = os.path.splitext(os.path.basename(f))[0]
    with open(f) as fh:
        checks[name] = json.load(fh)

result = {
    'timestamp': '$NOW_ISO',
    'epoch': $NOW,
    'overall': '$OVERALL',
    'cluster': '$CLUSTER',
    'region': '$REGION',
    'checks': checks
}

# Include domain if configured
domain = '$DOMAIN'
if domain:
    result['domain'] = domain

print(json.dumps(result, indent=2))
" 2>/dev/null)

# Write status file
echo "$FINAL_JSON" > "$STATUS_FILE"

# ---------------------------------------------------------------------------
# Output
# ---------------------------------------------------------------------------
if $JSON_ONLY; then
  echo "$FINAL_JSON"
  [[ "$OVERALL" == "healthy" ]] && exit 0 || exit 1
fi

# Build human-readable summary
if ! $QUIET && ! $JSON_ONLY; then
  if [[ "$OVERALL" == "healthy" ]]; then
    text="✅ ECS health check ($CLUSTER): all systems healthy"
  else
    issues=$(echo "$FINAL_JSON" | python3 -c "
import json, sys
data = json.load(sys.stdin)
issues = []
for name, check in data['checks'].items():
    if isinstance(check, dict) and check.get('status') not in ('healthy', 'scaled_down'):
        if 'issues' in check:
            issues.extend(check['issues'])
        elif 'error' in check:
            issues.append(f'{name}: {check[\"error\"]}')
        else:
            issues.append(f'{name}: {check.get(\"status\", \"unknown\")}')
print('; '.join(issues[:5]) if issues else '$OVERALL')
" 2>/dev/null)

    if [[ "$OVERALL" == "degraded" ]]; then
      text="⚠️  ECS health ($CLUSTER): DEGRADED — $issues"
    else
      text="🔴 ECS health ($CLUSTER): UNHEALTHY — $issues"
    fi
  fi

  echo "$text"
fi

# Write alert marker on failure (for external consumers)
if ! $QUIET && [[ "$OVERALL" != "healthy" ]]; then
  alert_file="${OUTDIR}/ecs-health-alert.json"
  echo "{\"timestamp\":\"$NOW_ISO\",\"cluster\":\"$CLUSTER\",\"overall\":\"$OVERALL\"}" > "$alert_file"

  # Auto-diagnose via log analyzer if available
  SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
  if [[ -x "$SCRIPT_DIR/cloudwatch-logs.sh" ]]; then
    LOG_ANALYSIS=$(ECS_CLUSTER="$CLUSTER" ECS_REGION="$REGION" ECS_SERVICES="${ECS_SERVICES:-}" \
      ECS_HEALTH_STATE="$STATUS_FILE" ECS_HEALTH_OUTDIR="$OUTDIR" \
      "$SCRIPT_DIR/cloudwatch-logs.sh" auto-diagnose --minutes 30 2>/dev/null)
    if [[ -n "$LOG_ANALYSIS" ]]; then
      python3 -c "
import json, sys
with open('$alert_file') as f:
    alert = json.load(f)
alert['log_analysis'] = sys.stdin.read().strip()
with open('$alert_file', 'w') as f:
    json.dump(alert, f, indent=2)
" <<< "$LOG_ANALYSIS" 2>/dev/null || true
    fi
  fi
fi

[[ "$OVERALL" == "healthy" ]] && exit 0 || exit 1
