#!/usr/bin/env bash
# =============================================================================
# ECS CloudWatch Log Puller & Analyzer
#
# Pull and analyze CloudWatch logs from ECS services.
# Integrates with ecs-health.sh for auto-diagnosis.
#
# Configuration (environment variables):
#   ECS_CLUSTER        (required) ECS cluster name
#   ECS_REGION         AWS region (default: us-east-1)
#   ECS_SERVICES       Comma-separated service names (auto-detect if unset)
#   ECS_HEALTH_STATE   Health state file path (default: ./data/ecs-health.json)
#   ECS_HEALTH_OUTDIR  Output directory (default: ./data/)
#   ECS_LOG_PATTERN    Log group pattern (default: /ecs/{service})
#
# Usage:
#   ./scripts/cloudwatch-logs.sh pull [service] [--minutes N] [--filter PATTERN]
#   ./scripts/cloudwatch-logs.sh errors [service] [--minutes N]
#   ./scripts/cloudwatch-logs.sh diagnose [service] [--minutes N]
#   ./scripts/cloudwatch-logs.sh restarts [service] [--minutes N]
#   ./scripts/cloudwatch-logs.sh summary [--minutes N]
#   ./scripts/cloudwatch-logs.sh auto-diagnose [--minutes N]
#
# Exit codes: 0 = success, 1 = issues found, 2 = script error
# =============================================================================
set -uo pipefail

# ---------------------------------------------------------------------------
# Configuration
# ---------------------------------------------------------------------------
if [[ -z "${ECS_CLUSTER:-}" ]]; then
  echo "ERROR: ECS_CLUSTER environment variable is required" >&2
  exit 2
fi

CLUSTER="${ECS_CLUSTER}"
REGION="${ECS_REGION:-us-east-1}"
OUTDIR="${ECS_HEALTH_OUTDIR:-./data}"
DATA_DIR="${OUTDIR}/logs"
HEALTH_FILE="${ECS_HEALTH_STATE:-${OUTDIR}/ecs-health.json}"
LOG_PATTERN="${ECS_LOG_PATTERN:-/ecs/{service}}"

COMMAND="" ; SERVICE="" ; MINUTES=60 ; FILTER=""
JSON_OUT=false ; LIMIT=200 ; VERBOSE=false

# ---------------------------------------------------------------------------
# Auto-detect services from cluster
# ---------------------------------------------------------------------------
detect_services() {
  aws ecs list-services --cluster "$CLUSTER" --region "$REGION" --output json 2>/dev/null \
    | python3 -c "
import json, sys
data = json.load(sys.stdin)
print(' '.join(a.split('/')[-1] for a in data.get('serviceArns', [])))
" 2>/dev/null
}

declare -a ALL_SERVICES_ARR
if [[ -n "${ECS_SERVICES:-}" ]]; then
  IFS=',' read -ra ALL_SERVICES_ARR <<< "$ECS_SERVICES"
else
  read -ra ALL_SERVICES_ARR <<< "$(detect_services)"
fi

if [[ ${#ALL_SERVICES_ARR[@]} -eq 0 ]]; then
  echo "ERROR: No services found in cluster '$CLUSTER'." >&2
  exit 2
fi

ALL_SERVICES="${ALL_SERVICES_ARR[*]}"

# Build log group mapping
declare -A LOG_GROUPS
for svc in ${ALL_SERVICES}; do
  LOG_GROUPS[$svc]="${LOG_PATTERN//\{service\}/$svc}"
done

# ---------------------------------------------------------------------------
# Parse arguments
# ---------------------------------------------------------------------------
parse_args() {
  COMMAND="${1:-}" ; shift || true
  while [[ $# -gt 0 ]]; do
    case "$1" in
      --minutes)  MINUTES="${2:-60}"; shift 2 ;;
      --filter)   FILTER="${2:-}"; shift 2 ;;
      --json)     JSON_OUT=true; shift ;;
      --limit)    LIMIT="${2:-200}"; shift 2 ;;
      --verbose)  VERBOSE=true; shift ;;
      -*)         echo "Unknown option: $1" >&2; exit 2 ;;
      *)          [[ -z "$SERVICE" ]] && SERVICE="$1"; shift ;;
    esac
  done
  [[ -z "$SERVICE" ]] && SERVICE="all"
}

resolve_services() {
  local svc="$1"
  if [[ "$svc" == "all" ]]; then echo "$ALL_SERVICES"; return; fi
  for s in ${ALL_SERVICES}; do
    [[ "$s" == "$svc" ]] && echo "$svc" && return
  done
  echo "Unknown service: $svc (available: $ALL_SERVICES)" >&2; exit 2
}

# ---------------------------------------------------------------------------
# Pull logs from a single service
# ---------------------------------------------------------------------------
pull_service_logs() {
  local svc="$1" log_group="${LOG_GROUPS[$1]}"
  local start_ms=$(( ($(date +%s) - MINUTES * 60) * 1000 ))
  local end_ms=$(( $(date +%s) * 1000 ))
  local cmd=(aws logs filter-log-events --log-group-name "$log_group"
    --region "$REGION" --start-time "$start_ms" --end-time "$end_ms"
    --limit "$LIMIT" --output json)
  [[ -n "$FILTER" ]] && cmd+=(--filter-pattern "$FILTER")
  "${cmd[@]}" 2>/dev/null || echo '{"events":[]}'
}

# ---------------------------------------------------------------------------
# cmd: pull
# ---------------------------------------------------------------------------
cmd_pull() {
  for svc in $(resolve_services "$SERVICE"); do
    local result; result=$(pull_service_logs "$svc")
    local lg="${LOG_GROUPS[$svc]}"
    if $JSON_OUT; then
      echo "$result" | python3 -c "
import json,sys; d=json.load(sys.stdin); e=d.get('events',[])
print(json.dumps({'service':'$svc','log_group':'$lg','count':len(e),'events':e},indent=2))"
    else
      local count; count=$(echo "$result" | python3 -c "import json,sys;print(len(json.load(sys.stdin).get('events',[])))" 2>/dev/null)
      echo "=== $svc ($lg) — $count events in last ${MINUTES}m ==="
      echo "$result" | python3 -c "
import json,sys,datetime
for e in json.load(sys.stdin).get('events',[]):
    ts=datetime.datetime.fromtimestamp(e['timestamp']/1000,datetime.timezone.utc).strftime('%H:%M:%S')
    m=e.get('message','').strip(); m=m[:500]+'...' if len(m)>500 else m
    print(f'  [{ts}] {m}')" 2>/dev/null
      echo ""
    fi
  done
}

# ---------------------------------------------------------------------------
# cmd: errors
# ---------------------------------------------------------------------------
cmd_errors() {
  local orig_filter="$FILTER" found=false
  FILTER='?ERROR ?error ?panic ?fatal ?FATAL ?"level":"error" ?Traceback ?Exception'
  for svc in $(resolve_services "$SERVICE"); do
    local result; result=$(pull_service_logs "$svc")
    local count; count=$(echo "$result" | python3 -c "import json,sys;print(len(json.load(sys.stdin).get('events',[])))" 2>/dev/null)
    if [[ "$count" -gt 0 ]]; then
      found=true
      if $JSON_OUT; then
        echo "$result" | python3 -c "
import json,sys; d=json.load(sys.stdin); e=d.get('events',[])
print(json.dumps({'service':'$svc','error_count':len(e),'events':e},indent=2))"
      else
        echo "🔴 $svc — $count error(s) in last ${MINUTES}m:"
        echo "$result" | python3 -c "
import json,sys,datetime
for e in json.load(sys.stdin).get('events',[]):
    ts=datetime.datetime.fromtimestamp(e['timestamp']/1000,datetime.timezone.utc).strftime('%H:%M:%S')
    m=e.get('message','').strip(); m=m[:300]+'...' if len(m)>300 else m
    print(f'  [{ts}] {m}')" 2>/dev/null
        echo ""
      fi
    else
      $JSON_OUT || echo "✅ $svc — no errors in last ${MINUTES}m"
    fi
  done
  FILTER="$orig_filter"
  [[ "$found" == "true" ]] && return 1 || return 0
}

# ---------------------------------------------------------------------------
# cmd: restarts
# ---------------------------------------------------------------------------
cmd_restarts() {
  local orig_filter="$FILTER"
  FILTER='?"Shutting down" ?"Starting" ?"started" ?"Uvicorn running" ?"Server started" ?"next start" ?"listening" ?"gracefully" ?"SIGTERM" ?"signal"'
  for svc in $(resolve_services "$SERVICE"); do
    pull_service_logs "$svc" | python3 -c "
import json,sys,datetime,re
events=json.load(sys.stdin).get('events',[])
up_re=[r'(?i)(starting|started|listening|running on|ready)',r'(?i)(uvicorn running|next start|gin-gonic)']
dn_re=[r'(?i)(shutting down|stopped|sigterm|graceful)']
ups,dns=[],[]
for e in events:
    m,ts=e.get('message','').strip(),datetime.datetime.fromtimestamp(e['timestamp']/1000,datetime.timezone.utc).strftime('%Y-%m-%d %H:%M:%S')
    if any(re.search(p,m) for p in up_re): ups.append((ts,m[:200]))
    if any(re.search(p,m) for p in dn_re): dns.append((ts,m[:200]))
print(f'=== $svc — {len(ups)} startups, {len(dns)} shutdowns (~{min(len(ups),len(dns))} restarts) in last ${MINUTES}m ===')
for ts,k,m in sorted([(t,'START',m) for t,m in ups]+[(t,'STOP',m) for t,m in dns]):
    print(f\"  {'🟢' if k=='START' else '🔴'} [{ts}] {m}\")
print()" 2>/dev/null
  done
  FILTER="$orig_filter"
}

# ---------------------------------------------------------------------------
# cmd: diagnose — deep analysis
# ---------------------------------------------------------------------------
cmd_diagnose() {
  mkdir -p "$DATA_DIR"
  local lg_json="{"
  local first=true
  for svc in $(resolve_services "$SERVICE"); do
    $first || lg_json+=","
    first=false
    lg_json+="\"$svc\":\"${LOG_GROUPS[$svc]}\""
  done
  lg_json+="}"

  CW_SERVICES="$(resolve_services "$SERVICE")" CW_REGION="$REGION" \
  CW_MINUTES="$MINUTES" CW_LIMIT="$LIMIT" CW_JSON_OUT="$JSON_OUT" \
  CW_VERBOSE="$VERBOSE" CW_LOG_GROUPS="$lg_json" python3 <<'PYEOF'
import json, sys, subprocess, re, datetime, os
from collections import Counter

services = os.environ["CW_SERVICES"].split()
region, minutes, limit = os.environ["CW_REGION"], int(os.environ["CW_MINUTES"]), int(os.environ["CW_LIMIT"])
json_out, verbose = os.environ["CW_JSON_OUT"]=="true", os.environ["CW_VERBOSE"]=="true"
log_groups = json.loads(os.environ["CW_LOG_GROUPS"])

ERROR_PATS = [
    (r'"level"\s*:\s*"error"',"structured_error"),(r'(?i)\bpanic\b',"panic"),
    (r'(?i)\bfatal\b',"fatal"),(r'(?i)\bERROR\b',"error"),
    (r'Traceback \(most recent',"python_traceback"),(r'(?i)exception',"exception"),
    (r'\| 5\d{2} \|',"http_5xx"),(r'(?i)timeout|timed out',"timeout"),
    (r'(?i)connection refused|connection reset',"connection_error"),
    (r'(?i)out of memory|OOM',"oom"),(r'(?i)permission denied|unauthorized|forbidden',"auth_error"),
]
RESTART_PATS = [(r'(?i)shutting down|stopped|sigterm',"shutdown"),
                (r'(?i)starting|started|listening|running on|ready',"startup")]
HTTP_RE = re.compile(r'\[GIN\]\s+\S+\s+-\s+\S+\s+\|\s+(\d{3})\s+\|')
HEALTH_RE = re.compile(r'(?:HEAD|GET)\s+"/health"')

_now = datetime.datetime.now(datetime.timezone.utc)
start_ms = int((_now - datetime.timedelta(minutes=minutes)).timestamp() * 1000)
end_ms = int(_now.timestamp() * 1000)
ts_fmt = lambda ms: datetime.datetime.fromtimestamp(ms/1000, datetime.timezone.utc).strftime("%H:%M:%S")
all_findings = {}

for svc in services:
    lg = log_groups.get(svc)
    if not lg: continue
    try:
        r = subprocess.run(["aws","logs","filter-log-events","--log-group-name",lg,
            "--region",region,"--start-time",str(start_ms),"--end-time",str(end_ms),
            "--limit",str(limit),"--output","json"], capture_output=True, text=True, timeout=30)
        events = json.loads(r.stdout).get("events", [])
    except Exception as e:
        all_findings[svc] = {"status":"error","error":str(e)}; continue
    if not events:
        all_findings[svc] = {"status":"no_logs","event_count":0}; continue

    f = {"event_count":len(events),"errors":[],"error_categories":Counter(),
         "restarts":[],"http_status_codes":Counter(),"unique_streams":set(),
         "time_range":{"start":ts_fmt(events[0]["timestamp"]),"end":ts_fmt(events[-1]["timestamp"])}}

    for evt in events:
        msg, ts, stream = evt.get("message","").strip(), evt.get("timestamp",0), evt.get("logStreamName","")
        f["unique_streams"].add(stream)
        is_hc = bool(HEALTH_RE.search(msg))
        if not is_hc:
            for pat, cat in ERROR_PATS:
                if re.search(pat, msg):
                    f["error_categories"][cat] += 1
                    f["errors"].append({"timestamp":ts_fmt(ts),"category":cat,"message":msg[:300],"stream":stream[-12:]})
                    break
        for pat, kind in RESTART_PATS:
            if re.search(pat, msg):
                f["restarts"].append({"timestamp":ts_fmt(ts),"kind":kind,"message":msg[:200]}); break
        hm = HTTP_RE.search(msg)
        if hm:
            code = hm.group(1)
            if not (code == "404" and is_hc): f["http_status_codes"][code] += 1

    ec = sum(f["error_categories"].values())
    rc = sum(1 for r in f["restarts"] if r["kind"] == "shutdown")
    ns = len(f["unique_streams"])
    nr = max(2, ns * max(1, minutes // 30))
    if any(c in f["error_categories"] for c in ("panic","fatal","oom")): f["status"] = "critical"
    elif ec > 10: f["status"] = "unhealthy"
    elif ec > 0 or rc > nr: f["status"] = "degraded"
    else: f["status"] = "healthy"

    f["unique_streams"] = ns
    f["error_categories"] = dict(f["error_categories"])
    f["http_status_codes"] = dict(f["http_status_codes"])
    seen, uniq = set(), []
    for err in f["errors"]:
        k = (err["category"], err["message"][:80])
        if k not in seen: seen.add(k); uniq.append(err)
    f["unique_error_count"], f["total_error_count"] = len(uniq), len(f["errors"])
    f["errors"] = uniq[:20] if not verbose else f["errors"][:50]
    all_findings[svc] = f

if json_out:
    print(json.dumps(all_findings, indent=2, default=str))
else:
    has_issues = False
    for svc, f in sorted(all_findings.items()):
        st = f.get("status","unknown")
        ic = {"healthy":"✅","degraded":"⚠️","unhealthy":"🔴","critical":"🚨","no_logs":"⚪","error":"❌"}.get(st,"❓")
        print(f"{ic} {svc} — {st} ({f.get('event_count',0)} events, {f.get('unique_streams',0)} streams)")
        if st in ("degraded","unhealthy","critical"): has_issues = True
        cats = f.get("error_categories",{})
        if cats:
            print(f"  Errors: {f.get('total_error_count',0)} total ({', '.join(f'{k}:{v}' for k,v in sorted(cats.items(),key=lambda x:-x[1]))})")
        http = f.get("http_status_codes",{})
        non200 = {k:v for k,v in http.items() if k != "200"}
        if non200:
            print(f"  HTTP: {sum(http.values())} total, non-200: {', '.join(f'{k}:{v}' for k,v in sorted(non200.items()))}")
        rs = f.get("restarts",[])
        sd, su = sum(1 for r in rs if r["kind"]=="shutdown"), sum(1 for r in rs if r["kind"]=="startup")
        if sd or su: print(f"  Restarts: {sd} shutdowns, {su} startups")
        errs = f.get("errors",[])
        if errs:
            print(f"  Sample errors ({min(len(errs),5)} of {f.get('unique_error_count',len(errs))} unique):")
            for e in errs[:5]:
                m = e["message"][:120]+"..." if len(e["message"])>120 else e["message"]
                print(f"    [{e['timestamp']}] ({e['category']}) {m}")
        print()
    if has_issues: sys.exit(1)
PYEOF
}

# ---------------------------------------------------------------------------
# cmd: summary
# ---------------------------------------------------------------------------
cmd_summary() { SERVICE="all"; cmd_diagnose; }

# ---------------------------------------------------------------------------
# cmd: auto-diagnose — parse health state, then run diagnose on failing svcs
# ---------------------------------------------------------------------------
cmd_auto_diagnose() {
  if [[ ! -f "$HEALTH_FILE" ]]; then
    echo "No health status file found at $HEALTH_FILE — run ecs-health.sh first."
    exit 2
  fi

  # Parse health file to find failing services and print header
  local failing_svcs
  failing_svcs=$(python3 -c "
import json, sys
with open('$HEALTH_FILE') as f: health = json.load(f)
overall = health.get('overall','unknown')
checks = health.get('checks',{})
print(f'Production Status: {overall} (from {health.get(\"timestamp\",\"unknown\")})', file=sys.stderr)
print(file=sys.stderr)

svcs, issues = [], []
ecs = checks.get('ecs_services',{})
if ecs.get('status') not in ('healthy',):
    for n, d in ecs.get('services',{}).items():
        h = d.get('health','unknown')
        if h in ('degraded','unhealthy','failed'):
            svcs.append(n)
            issues.append(f'{n}: {h} (desired={d.get(\"desired\")}, running={d.get(\"running\")})')
for key in ('domain_http',):
    c = checks.get(key,{})
    if c.get('status') != 'healthy':
        issues.append(f'{key}: {c.get(\"status\")} (code={c.get(\"http_code\")})')
for name, c in checks.items():
    if name.startswith('http_') and isinstance(c,dict) and c.get('status') != 'healthy':
        issues.append(f'{name}: {c.get(\"status\")}')
alb = checks.get('alb_targets',{})
if alb.get('status') not in ('healthy',):
    for tn, td in alb.get('target_groups',{}).items():
        if td.get('status') not in ('healthy','empty'):
            issues.append(f'ALB {tn}: {td.get(\"status\")}')

if not svcs and overall == 'healthy':
    print('', file=sys.stderr)
    print('✅ All services healthy — no log analysis needed.', file=sys.stderr)
    sys.exit(0)

if issues:
    print('Issues detected:', file=sys.stderr)
    for i in issues: print(f'  ⚠️  {i}', file=sys.stderr)
    print(file=sys.stderr)

# If we couldn't identify specific services, check those with desired > 0
if not svcs:
    ecs_svcs = ecs.get('services',{})
    svcs = [n for n,d in ecs_svcs.items() if d.get('desired',0) > 0]

# Output service names to stdout for bash to consume
print(' '.join(svcs) if svcs else '')
" 2>&1 1>/tmp/ecs-ad-svcs.$$)

  # Print the header/issues (stderr was redirected to stdout above)
  echo "$failing_svcs"

  # Read the services list
  local svc_list
  svc_list=$(cat /tmp/ecs-ad-svcs.$$ 2>/dev/null)
  rm -f /tmp/ecs-ad-svcs.$$

  if [[ -z "$svc_list" ]]; then
    return 0
  fi

  echo "Analyzing logs for: $svc_list (last ${MINUTES}m)"
  echo "$(printf '=%.0s' {1..60})"
  echo ""

  # Diagnose each failing service using the existing diagnose command
  for svc in $svc_list; do
    # Check if this service is in our known list
    local known=false
    for s in ${ALL_SERVICES}; do
      [[ "$s" == "$svc" ]] && known=true && break
    done
    if $known; then
      SERVICE="$svc"
      LIMIT=300
      cmd_diagnose
    else
      echo "⚠️  $svc — not in detected service list, skipping"
    fi
  done
}

# ---------------------------------------------------------------------------
# Usage
# ---------------------------------------------------------------------------
usage() {
  cat <<EOF
ECS CloudWatch Log Puller & Analyzer

Usage:
  $0 pull [service] [--minutes N] [--filter PAT]   Pull raw logs
  $0 errors [service] [--minutes N]                  Show errors only
  $0 diagnose [service] [--minutes N]                Deep log analysis
  $0 restarts [service] [--minutes N]                Detect restarts
  $0 summary [--minutes N]                           Summary across all
  $0 auto-diagnose [--minutes N]                     Health state + analyze

Services: auto-detected from cluster (or set ECS_SERVICES). "all" = default.
Options:  --minutes N (60), --json, --limit N (200), --verbose

Environment:
  ECS_CLUSTER       (required) ECS cluster name
  ECS_REGION        AWS region (default: us-east-1)
  ECS_SERVICES      Comma-separated service names
  ECS_HEALTH_STATE  Health state file path
  ECS_HEALTH_OUTDIR Output directory
  ECS_LOG_PATTERN   Log group pattern (default: /ecs/{service})
EOF
}

# ---------------------------------------------------------------------------
# Main
# ---------------------------------------------------------------------------
parse_args "$@"
mkdir -p "$DATA_DIR"
case "$COMMAND" in
  pull)           cmd_pull ;;
  errors)         cmd_errors ;;
  diagnose)       cmd_diagnose ;;
  restarts)       cmd_restarts ;;
  summary)        cmd_summary ;;
  auto-diagnose)  cmd_auto_diagnose ;;
  help|-h|--help) usage ;;
  *)              echo "Unknown command: ${COMMAND:-<none>}" >&2; usage >&2; exit 2 ;;
esac
