#!/bin/bash
# autonomous-tasks 错误恢复脚本
# 自动检测并尝试恢复常见错误

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SKILL_DIR="$(dirname "$SCRIPT_DIR")"
BACKUP_DIR="$SKILL_DIR/.backups"

echo "=== Autonomous-Tasks 错误恢复 ==="
echo "执行时间: $(date '+%Y-%m-%d %H:%M:%S')"
echo ""

# 确保备份目录存在
mkdir -p "$BACKUP_DIR"

RECOVERED=0
FAILED=0

# 1. 检查并恢复任务日志
echo "📋 检查任务日志..."
TASK_LOG="$SKILL_DIR/memory/tasks-log.md"
if [ -f "$TASK_LOG" ]; then
    # 备份当前日志
    cp "$TASK_LOG" "$BACKUP_DIR/tasks-log-$(date +%Y%m%d-%H%M%S).bak"
    
    # 检查格式完整性
    if grep -q "^- \[ \]" "$TASK_LOG"; then
        echo "  ⚠️  发现未完成任务标记，清理中..."
        # 将未完成标记转为注释
        sed -i '' 's/^- \[ \] /# 未完成: /g' "$TASK_LOG"
        ((RECOVERED++))
    fi
    
    # 检查是否有损坏行
    if grep -qE "^- ✅.*)$" "$TASK_LOG"; then
        echo "  ✅ 任务日志格式正常"
    else
        echo "  ⚠️  任务日志格式可能有问题"
    fi
else
    echo "  ❌ 任务日志不存在，尝试创建..."
    echo "# 任务日志 (append-only)" > "$TASK_LOG"
    ((RECOVERED++))
fi

# 2. 检查并恢复 AUTONOMOUS.md
echo ""
echo "🎯 检查目标文件..."
AUTONOMOUS="$SKILL_DIR/AUTONOMOUS.md"
if [ -f "$AUTONOMOUS" ]; then
    # 备份
    cp "$AUTONOMOUS" "$BACKUP_DIR/autonomous-$(date +%Y%m%d-%H%M%S).bak"
    
    # 检查必要章节
    if ! grep -q "## 长期目标" "$AUTONOMOUS"; then
        echo "  ⚠️  缺少长期目标章节"
        ((FAILED++))
    else
        echo "  ✅ 目标文件完整"
    fi
else
    echo "  ❌ 目标文件不存在"
    ((FAILED++))
fi

# 3. 检查目录结构
echo ""
echo "📂 检查目录..."
REQUIRED_DIRS=("research" "drafts" "apps" "scripts" "memory")
for dir in "${REQUIRED_DIRS[@]}"; do
    if [ ! -d "$SKILL_DIR/$dir" ]; then
        echo "  ⚠️  $dir/ 不存在，创建中..."
        mkdir -p "$SKILL_DIR/$dir"
        ((RECOVERED++))
    fi
done
echo "  ✅ 目录结构完整"

# 4. 清理过期备份
echo ""
echo "🧹 清理过期备份 (保留最近 10 个)..."
BACKUP_COUNT=$(ls -1 "$BACKUP_DIR" 2>/dev/null | wc -l)
if [ "$BACKUP_COUNT" -gt 10 ]; then
    ls -t "$BACKUP_DIR" | tail -$((BACKUP_COUNT - 10)) | while read f; do
        rm "$BACKUP_DIR/$f"
    done
    echo "  ✅ 已清理 $((BACKUP_COUNT - 10)) 个过期备份"
else
    echo "  ✅ 备份数量正常 ($BACKUP_COUNT)"
fi

# 总结
echo ""
echo "=== 恢复总结 ==="
echo "已恢复: $RECOVERED 项"
echo "失败: $FAILED 项"

if [ $RECOVERED -gt 0 ]; then
    echo ""
    echo "📝 备份位置: $BACKUP_DIR"
fi

if [ $FAILED -eq 0 ]; then
    echo ""
    echo "🎉 系统已恢复正常"
    exit 0
else
    echo ""
    echo "⚠️  需要手动干预"
    exit 1
fi
