# Task Types — Catalog & Execution Strategies

This file defines the recognized task types, how to classify user tasks, and what execution
strategies to apply (in order of preference).

> **Context:** In the task-runner skill, type classification happens during INTAKE (if clear) or
> at the start of DISPATCHER execution (if `type = "unknown"`). See `SKILL.md` for the two-mode
> workflow and `references/queue-schema.md` for the full task JSON schema.

---

## Classification Rules

When parsing a user task, assign the **first matching type** from this list:

1. **messaging** — task involves sending a message to a person or channel
2. **reminder-scheduling** — task is a time-based trigger or recurring action
3. **code-execution** — task involves running a script, command, or program
4. **file-creation** — task involves creating, editing, or saving a file
5. **agent-delegation** — task requires complex multi-step work best handed off
6. **info-lookup** — task requires finding information from the web or a known source
7. **unknown** — doesn't clearly fit any of the above

**When in doubt, prefer `info-lookup`.** It's the lowest-risk default and often the right answer.

> **Note on agent-delegation:** In the new design, all tasks are executed by subagents spawned by
> the DISPATCHER. `agent-delegation` as a task type means the task itself is complex enough to
> warrant a *dedicated* subagent with a specific multi-step prompt, rather than a simple tool call.

---

## Type: `info-lookup`

**Description:** The task requires finding, retrieving, or synthesizing information.

**Examples:**
- "Find the current price of gold"
- "What is the capital of [country]?"
- "Look up the documentation for [tool]"
- "Summarize recent news about [topic]"
- "Check if [website] is down"

**Execution Strategies (in order):**

| Priority | Strategy | Tool | When to Use |
|----------|----------|------|-------------|
| 1 | Web search | `web_search` | Most info-lookup tasks |
| 2 | Fetch specific URL | `web_fetch` | User provides a URL or search returns a specific page |
| 3 | Browser navigation | `browser` | Page requires JavaScript to load or requires interaction |
| 4 | Ask user for source | (ask) | After 3 failed attempts; user may know the right URL |

**Verification:** See `verification-guide.md` → info-lookup section

**Common failure modes:**
- Search returns ads or SEO spam → try `web_fetch` directly to authoritative source
- Page is paywalled → note in blocked_reason; tell user which site and what access is needed
- Query too vague → reformulate with more specific terms before retry

---

## Type: `file-creation`

**Description:** The task involves creating a new file, writing content to disk, or editing an existing file.

**Examples:**
- "Create a markdown file with X content"
- "Write a report and save it to ~/reports/"
- "Update my config file to add X"
- "Generate a template for Y and save it"

**Execution Strategies (in order):**

| Priority | Strategy | Tool | When to Use |
|----------|----------|------|-------------|
| 1 | Direct write | `write` tool | New file or full overwrite |
| 2 | Surgical edit | `edit` tool | Modifying specific section of existing file |
| 3 | Exec-based write | `exec` (echo/cat) | When content is generated by a command |
| 4 | Provide draft in chat | (message) | If path is inaccessible; give user content to save |

**Verification:** File exists at expected path; non-empty; content matches intent

**Common failure modes:**
- Path doesn't exist → create parent directories first with `exec mkdir -p`
- Permission denied → report in blocked_reason with path and required permission
- Content was generated but wrong → retry with corrected content (counts as 1 retry)

---

## Type: `code-execution`

**Description:** The task requires running a script, shell command, or code snippet.

**Examples:**
- "Run this Python script"
- "Execute the build script"
- "Install the npm dependencies"
- "Check if the service is running"
- "Run the tests"

**Execution Strategies (in order):**

| Priority | Strategy | Tool | When to Use |
|----------|----------|------|-------------|
| 1 | Direct exec | `exec` tool | Standard shell commands |
| 2 | Exec with error handling | `exec` with `|| true` | Recoverable errors expected |
| 3 | Exec in PTY mode | `exec` with `pty=true` | Interactive commands or TTY-required programs |
| 4 | Write script then exec | `write` + `exec` | Complex multi-line script |

**Verification:** Exit code = 0; expected output present in stdout; no error indicators in stderr

**Common failure modes:**
- Command not found → check if tool is installed; note in blocked_reason
- Permission denied → try with elevated flag if appropriate; otherwise block
- Script errors → capture stderr; attempt to fix and retry if error is clear
- Timeout → increase timeout or break into smaller commands

---

## Type: `agent-delegation`

**Description:** The task is complex, multi-step, or benefits from isolation in a separate agent session.

**Examples:**
- "Have a sub-agent research and write a full report on X"
- "Delegate the data analysis to a fresh agent"
- "Spawn an agent to handle the entire Y workflow"

**Execution Strategies (in order):**

| Priority | Strategy | Tool | When to Use |
|----------|----------|------|-------------|
| 1 | Spawn sub-agent | `sessions_spawn` (or equivalent) | Full delegation to independent session |
| 2 | Inline sub-task | Run inline with clear role boundaries | If spawning unavailable |
| 3 | Report limitation | (message) | If neither delegation method is available |

**Verification:** Sub-agent reports completion; deliverable received and non-empty; no error state

**Common failure modes:**
- Session spawn not available → fall back to inline execution
- Sub-agent times out → note in blocked_reason; ask user to rerun
- Deliverable not returned → check sub-agent output; retry if output file expected

---

## Type: `reminder-scheduling`

**Description:** The task involves setting a time-based trigger, recurring job, or scheduled action.

**Examples:**
- "Remind me at 3pm to call the dentist"
- "Set a weekly reminder to review my tasks"
- "Schedule this command to run every Monday morning"

**Execution Strategies (in order):**

| Priority | Strategy | Tool | When to Use |
|----------|----------|------|-------------|
| 1 | Cron tool | cron (if available) | Recurring or specific-time reminders |
| 2 | Write reminder file | `write` to `~/reminders/` | Cron not available; user can check later |
| 3 | Notify user to schedule manually | (message) | Neither cron nor file write available |

**Verification:** Cron job registered (confirm with `crontab -l`); OR reminder file exists with correct content

**Common failure modes:**
- Cron not available in environment → write to reminder file and notify user
- Time zone ambiguous → ask user to confirm time zone before scheduling
- Recurring schedule unclear → parse best-effort; confirm with user in notification

---

## Type: `messaging`

**Description:** The task involves sending a message to a person, channel, or service.

**Examples:**
- "Send a message to #general on Slack"
- "Email the team about X"
- "Post a tweet about Y"
- "DM user@example.com with Z"

**Execution Strategies (in order):**

| Priority | Strategy | Tool | When to Use |
|----------|----------|------|-------------|
| 1 | Message tool | `message` | Configured channels (Telegram, Discord, etc.) |
| 2 | Channel-specific fallback | channel's API via `exec` curl | If message tool unavailable for that channel |
| 3 | Report limitation | (message to user) | Channel/credentials not configured |

**Verification:** Message delivery confirmed (no send error); message appears in channel if checkable

**Common failure modes:**
- Channel not configured → block with instructions to configure the channel
- Credentials expired → block with instructions to re-authenticate
- Rate limited → wait and retry (counts as retry); note delay in notification

---

## Type: `unknown`

**Description:** The task doesn't clearly fit any of the above types.

**Examples:**
- "Do the thing we talked about last week" (ambiguous)
- "Take care of the project" (too vague)
- "Handle the situation" (no clear action)

**Execution Strategies (in order):**

| Priority | Strategy | Tool | When to Use |
|----------|----------|------|-------------|
| 1 | Web search | `web_search` | Try to gather context that helps classify |
| 2 | Re-classify | (internal) | After web search, attempt to assign a real type |
| 3 | Ask user | (message) | ONE clarifying question: "What should I do for [task]?" |
| 4 | Block with clarification request | blocked status | If user doesn't respond or answer is still unclear |

**Note:** After asking the user one clarifying question, wait for a response before counting a retry. If the user doesn't respond within the session, mark blocked with `user_action_required = "Please clarify what action you want for [task description]"`.

---

## Strategy Selection Decision Tree

```
Is it about sending a message to someone?
  YES → messaging
  NO ↓

Is it time-based or recurring?
  YES → reminder-scheduling
  NO ↓

Does it involve running code/commands?
  YES → code-execution
  NO ↓

Does it involve creating or editing files?
  YES → file-creation
  NO ↓

Is it complex enough to warrant a separate agent?
  YES → agent-delegation
  NO ↓

Does it require finding or fetching information?
  YES → info-lookup
  NO ↓

unknown → try web_search → re-classify → ask user
```

---

## Retry Strategy Rotation

When a strategy fails, pick the NEXT one in the priority list. Track tried strategies in `strategies_tried`:

```json
"strategies_tried": [
  {"strategy": "web_search", "tool": "web_search", "result": "no relevant results"},
  {"strategy": "web_fetch", "tool": "web_fetch", "url": "https://example.com/data", "result": "404 not found"}
]
```

**Never retry the same strategy twice in a row** (unless the error suggests a transient failure like a timeout, in which case one retry of the same strategy is permitted).
