#!/usr/bin/env python3
"""Compliance report generator.

Generates HTML reports from the GRC database for a given framework.
Uses Jinja2 templates for formatting.

Usage:
    python3 generate_report.py --framework <slug> [--format html] [--output-dir <path>] [--db-path <path>]

Exit codes:
    0 = success
    1 = error
"""

import argparse
import json
import os
import sqlite3
import sys
from datetime import datetime
from html import escape

DEFAULT_DB = os.path.expanduser("~/.openclaw/grc/compliance.sqlite")
ASSETS_DIR = os.path.join(os.path.dirname(os.path.abspath(__file__)), "..", "assets")
DEFAULT_OUTPUT = os.path.expanduser("~/.openclaw/grc/reports")

# Inline template for when Jinja2 is not available or for simple HTML generation
HTML_TEMPLATE = """<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Compliance Report - {framework_name}</title>
    <style>
        body {{ font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif; margin: 40px; color: #333; }}
        h1 {{ color: #1a1a2e; border-bottom: 2px solid #16213e; padding-bottom: 10px; }}
        h2 {{ color: #16213e; margin-top: 30px; }}
        .score {{ font-size: 48px; font-weight: bold; text-align: center; padding: 20px; border-radius: 8px; margin: 20px 0; }}
        .score.excellent {{ background: #d4edda; color: #155724; }}
        .score.good {{ background: #d1ecf1; color: #0c5460; }}
        .score.fair {{ background: #fff3cd; color: #856404; }}
        .score.needs-attention {{ background: #f8d7da; color: #721c24; }}
        .score.critical {{ background: #f5c6cb; color: #721c24; }}
        table {{ width: 100%; border-collapse: collapse; margin: 15px 0; }}
        th, td {{ padding: 10px 12px; text-align: left; border-bottom: 1px solid #ddd; }}
        th {{ background: #f8f9fa; font-weight: 600; }}
        .status-complete {{ color: #28a745; }}
        .status-in_progress {{ color: #ffc107; }}
        .status-not_started {{ color: #dc3545; }}
        .meta {{ color: #666; font-size: 0.9em; }}
        .summary {{ display: flex; gap: 20px; flex-wrap: wrap; }}
        .summary-card {{ background: #f8f9fa; padding: 15px 20px; border-radius: 8px; flex: 1; min-width: 150px; }}
        .summary-card h3 {{ margin: 0 0 5px 0; font-size: 0.85em; color: #666; text-transform: uppercase; }}
        .summary-card .value {{ font-size: 24px; font-weight: bold; }}
        .footer {{ margin-top: 40px; padding-top: 20px; border-top: 1px solid #ddd; color: #999; font-size: 0.8em; }}
    </style>
</head>
<body>
    <h1>Compliance Report: {framework_name}</h1>
    <p class="meta">Generated: {generated_at} | Framework: {framework_slug}</p>

    <div class="score {score_class}">
        {score}/100 — {score_label}
    </div>

    <div class="summary">
        <div class="summary-card"><h3>Total Controls</h3><div class="value">{total_controls}</div></div>
        <div class="summary-card"><h3>Complete</h3><div class="value status-complete">{complete_controls}</div></div>
        <div class="summary-card"><h3>In Progress</h3><div class="value status-in_progress">{in_progress_controls}</div></div>
        <div class="summary-card"><h3>Not Started</h3><div class="value status-not_started">{not_started_controls}</div></div>
        <div class="summary-card"><h3>Evidence Items</h3><div class="value">{evidence_count}</div></div>
        <div class="summary-card"><h3>Open Risks</h3><div class="value">{risk_count}</div></div>
    </div>

    <h2>Controls</h2>
    <table>
        <thead>
            <tr><th>ID</th><th>Title</th><th>Category</th><th>Priority</th><th>Status</th><th>Assignee</th></tr>
        </thead>
        <tbody>
            {control_rows}
        </tbody>
    </table>

    <h2>Evidence</h2>
    <table>
        <thead>
            <tr><th>Title</th><th>Type</th><th>Status</th><th>Valid Until</th></tr>
        </thead>
        <tbody>
            {evidence_rows}
        </tbody>
    </table>

    <h2>Risks</h2>
    <table>
        <thead>
            <tr><th>Title</th><th>Score</th><th>Treatment</th><th>Status</th></tr>
        </thead>
        <tbody>
            {risk_rows}
        </tbody>
    </table>

    <div class="footer">
        Generated by GRC Compliance Skill v1.0 | OpenClaw
    </div>
</body>
</html>"""


def get_score_class(score):
    """Map score to CSS class."""
    if score >= 90:
        return "excellent"
    elif score >= 75:
        return "good"
    elif score >= 60:
        return "fair"
    elif score >= 50:
        return "needs-attention"
    return "critical"


def get_score_label(score):
    """Map score to label."""
    if score >= 90:
        return "Excellent"
    elif score >= 75:
        return "Good"
    elif score >= 60:
        return "Fair"
    elif score >= 50:
        return "Needs Attention"
    return "Critical"


def generate_report(db_path, framework_slug, output_dir):
    """Generate an HTML compliance report."""
    if not os.path.exists(db_path):
        return {"status": "error", "message": f"Database not found: {db_path}"}

    conn = sqlite3.connect(db_path)
    conn.row_factory = sqlite3.Row

    # Get framework
    fw = conn.execute("SELECT * FROM frameworks WHERE slug = ? AND status = 'active'",
                       (framework_slug,)).fetchone()
    if not fw:
        conn.close()
        return {"status": "error", "message": f"Framework not found or inactive: {framework_slug}"}

    framework_name = fw["name"]
    framework_id = fw["id"]

    # Get controls
    controls = conn.execute(
        "SELECT * FROM controls WHERE framework_id = ? ORDER BY control_id",
        (framework_id,)
    ).fetchall()

    # Count by status
    status_counts = {"complete": 0, "in_progress": 0, "not_started": 0, "other": 0}
    for c in controls:
        s = c["status"]
        if s in status_counts:
            status_counts[s] += 1
        else:
            status_counts["other"] += 1

    # Get evidence count
    evidence_count = conn.execute("SELECT COUNT(*) as cnt FROM evidence").fetchone()["cnt"]
    evidence_rows_data = conn.execute(
        "SELECT title, type, status, valid_until FROM evidence ORDER BY valid_until"
    ).fetchall()

    # Get risk count
    risk_count = conn.execute("SELECT COUNT(*) as cnt FROM risks WHERE status = 'open'").fetchone()["cnt"]
    risk_rows_data = conn.execute(
        "SELECT title, likelihood * impact as score, treatment, status FROM risks ORDER BY likelihood * impact DESC"
    ).fetchall()

    # Calculate score (simple: complete/total * 100)
    total = len(controls)
    if total > 0:
        score = round((status_counts["complete"] / total) * 100, 1)
    else:
        score = 0

    # Check for stored score
    stored = conn.execute(
        "SELECT score FROM compliance_scores WHERE framework_slug = ? ORDER BY calculated_at DESC LIMIT 1",
        (framework_slug,)
    ).fetchone()
    if stored:
        score = stored["score"]

    conn.close()

    # Build HTML rows
    control_html = ""
    for c in controls:
        status_class = f"status-{escape(c['status'] or '')}"
        control_html += f'<tr><td>{escape(c["control_id"] or "")}</td><td>{escape(c["title"] or "")}</td>'
        control_html += f'<td>{escape(c["category"] or "")}</td><td>P{c["priority"]}</td>'
        control_html += f'<td class="{status_class}">{escape(c["status"] or "")}</td>'
        control_html += f'<td>{escape(c["assignee"] or "")}</td></tr>\n'

    evidence_html = ""
    for e in evidence_rows_data:
        evidence_html += f'<tr><td>{escape(e["title"] or "")}</td><td>{escape(e["type"] or "")}</td>'
        evidence_html += f'<td>{escape(e["status"] or "")}</td><td>{escape(e["valid_until"] or "N/A")}</td></tr>\n'

    risk_html = ""
    for r in risk_rows_data:
        risk_html += f'<tr><td>{escape(r["title"] or "")}</td><td>{escape(str(r["score"]) if r["score"] else "N/A")}</td>'
        risk_html += f'<td>{escape(r["treatment"] or "")}</td><td>{escape(r["status"] or "")}</td></tr>\n'

    if not control_html:
        control_html = '<tr><td colspan="6">No controls found</td></tr>'
    if not evidence_html:
        evidence_html = '<tr><td colspan="4">No evidence records</td></tr>'
    if not risk_html:
        risk_html = '<tr><td colspan="4">No risks registered</td></tr>'

    # Generate HTML
    html = HTML_TEMPLATE.format(
        framework_name=escape(framework_name),
        framework_slug=escape(framework_slug),
        generated_at=datetime.now().strftime("%Y-%m-%d %H:%M"),
        score=score,
        score_class=get_score_class(score),
        score_label=get_score_label(score),
        total_controls=total,
        complete_controls=status_counts["complete"],
        in_progress_controls=status_counts["in_progress"],
        not_started_controls=status_counts["not_started"],
        evidence_count=evidence_count,
        risk_count=risk_count,
        control_rows=control_html,
        evidence_rows=evidence_html,
        risk_rows=risk_html
    )

    # Write file
    os.makedirs(output_dir, exist_ok=True)
    filename = f"{framework_slug}-report-{datetime.now().strftime('%Y-%m-%d')}.html"
    filepath = os.path.join(output_dir, filename)
    with open(filepath, "w") as f:
        f.write(html)

    return {
        "status": "generated",
        "path": filepath,
        "format": "html",
        "framework": framework_slug,
        "controls": total,
        "score": score
    }


def main():
    parser = argparse.ArgumentParser(description="Generate compliance report")
    parser.add_argument("--framework", required=True, help="Framework slug")
    parser.add_argument("--format", choices=["html"], default="html", help="Output format")
    parser.add_argument("--output-dir", dest="output_dir", default=DEFAULT_OUTPUT, help="Output directory")
    parser.add_argument("--db-path", dest="db_path", default=DEFAULT_DB, help="Database path")
    args = parser.parse_args()

    result = generate_report(args.db_path, args.framework, args.output_dir)

    print(json.dumps(result, indent=2))
    if result["status"] == "error":
        sys.exit(1)


if __name__ == "__main__":
    main()
