#!/usr/bin/env bash

# Speech is Cheap (SIC) CLI Wrapper for OpenClaw
# Documentation: https://docs.speechischeap.com

API_BASE="https://api.speechischeap.com/v2"
UPLOAD_BASE="https://upload.speechischeap.com/v2"

# 1. Check for API Key
if [ -z "$SIC_API_KEY" ]; then
    echo "❌ Error: SIC_API_KEY environment variable is not set."
    echo "--------------------------------------------------------"
    echo "To use this skill, you need a Speech is Cheap API key."
    echo "1. Sign up at: https://speechischeap.com"
    echo "2. Copy your key."
    echo "3. Run: export SIC_API_KEY='your_key_here'"
    echo "--------------------------------------------------------"
    exit 1
fi

COMMAND=$1
shift

case "$COMMAND" in
    transcribe)
        URL=""
        FILE=""
        SPEAKERS=false
        WORDS=false
        LABELS=false
        STREAM=false
        PRIVATE=false
        LANGUAGE=""
        CONFIDENCE=0.5
        FORMAT=""
        WEBHOOK=""
        SEGMENT_DURATION=30

        while [[ "$#" -gt 0 ]]; do
            case $1 in
                --url) URL="$2"; shift ;;
                --file) FILE="$2"; shift ;;
                --speakers) SPEAKERS=true ;;
                --words) WORDS=true ;;
                --labels) LABELS=true ;;
                --stream) STREAM=true ;;
                --private) PRIVATE=true ;;
                --language) LANGUAGE="$2"; shift ;;
                --confidence) CONFIDENCE="$2"; shift ;;
                --format) FORMAT="$2"; shift ;;
                --webhook) WEBHOOK="$2"; shift ;;
                --segment-duration) SEGMENT_DURATION="$2"; shift ;;
                *) echo "Unknown parameter: $1"; exit 1 ;;
            esac
            shift
        done

        if [ -n "$URL" ]; then
            # Transcribe via URL - build JSON with input_url included
            JSON_DATA=$(cat <<EOF
{
  "input_url": "$URL",
  "can_parse_speakers": $SPEAKERS,
  "can_parse_words": $WORDS,
  "can_label_audio": $LABELS,
  "can_stream_output": $STREAM,
  "is_private": $PRIVATE,
  "language": "$LANGUAGE",
  "minimum_confidence": $CONFIDENCE,
  "output_format": "$FORMAT",
  "webhook_url": "$WEBHOOK",
  "segment_duration": $SEGMENT_DURATION
}
EOF
)
            curl -s -X POST "$API_BASE/jobs/" \
                -H "Authorization: Bearer $SIC_API_KEY" \
                -H "Content-Type: application/json" \
                -d "$JSON_DATA"
        elif [ -n "$FILE" ]; then
            # Transcribe via Upload
            curl -s -X POST "$UPLOAD_BASE/jobs/" \
                -H "Authorization: Bearer $SIC_API_KEY" \
                -F "input_file=@$FILE" \
                -F "can_parse_speakers=$SPEAKERS" \
                -F "can_parse_words=$WORDS" \
                -F "can_label_audio=$LABELS" \
                -F "can_stream_output=$STREAM" \
                -F "is_private=$PRIVATE" \
                -F "language=$LANGUAGE" \
                -F "minimum_confidence=$CONFIDENCE" \
                -F "output_format=$FORMAT" \
                -F "webhook_url=$WEBHOOK" \
                -F "segment_duration=$SEGMENT_DURATION"
        else
            echo "Usage: sic transcribe --url <url> OR --file <path>"
            exit 1
        fi
        ;;

    status)
        JOB_ID=$1
        if [ -z "$JOB_ID" ]; then
            echo "Usage: sic status <job_id>"
            exit 1
        fi
        curl -s -X GET "$API_BASE/jobs/$JOB_ID" \
            -H "Authorization: Bearer $SIC_API_KEY"
        ;;

    help|*)
        echo "Speech is Cheap (SIC) Skill"
        echo "Usage:"
        echo "  sic transcribe --url <url> [options]"
        echo "  sic transcribe --file <path> [options]"
        echo ""
        echo "  Options:"
        echo "    --speakers             Enable speaker diarization"
        echo "    --words                Enable word-level timestamps"
        echo "    --labels               Enable audio labeling"
        echo "    --stream               Enable streaming output"
        echo "    --private              Enable private mode"
        echo "    --language <code>      Set language (e.g., 'en', 'fr')"
        echo "    --confidence <float>   Set minimum confidence (default 0.5)"
        echo "    --format <fmt>         Set output format (srt, vtt, webvtt)"
        echo "    --webhook <url>        Set webhook URL for async results"
        echo "    --segment-duration <n> Set segment duration in seconds (default 30)"
        echo "  sic status <job_id>"
        ;;
esac
