const fs = require("fs");
const path = require("path");

const WORKSPACE_ROOT = path.resolve(__dirname, "../..");
const PENDING_TASK_PATH = path.join(WORKSPACE_ROOT, "memory/evolution/pending_skill_task.json");

function ensureDir(dirPath) {
  if (!fs.existsSync(dirPath)) {
    fs.mkdirSync(dirPath, { recursive: true });
  }
}

function sanitizeSlug(value) {
  return String(value || "")
    .toLowerCase()
    .replace(/[^a-z0-9]+/g, "-")
    .replace(/^-+|-+$/g, "")
    .slice(0, 40);
}

function renderSkillMarkdown({ skillName, paper }) {
  const title = String(paper.title || "Untitled Paper");
  const categories = String(paper.categories || "research");
  const paperKey = String(paper.paper_key || "unknown-paper-key");
  const abstract = String(paper.abstract || "").trim();
  const abstractSnippet = abstract ? abstract.slice(0, 500) : "No abstract provided.";
  const description = `Learned from arXiv paper ${title}. Use this skill to scaffold Node.js experiments based on the paper method.`;

  return `---
name: ${skillName}
description: ${JSON.stringify(description)}
metadata: '{"openclaw":{"requires":{"bins":["node"]}}}'
---

# ${skillName}

## Source
- Paper key: ${paperKey}
- Title: ${title}
- Categories: ${categories}

## Learned insight
${abstractSnippet}

## Node.js implementation entry
\`node {baseDir}/scripts/run.js\`
`;
}

function renderNodeRunner({ paper }) {
  const title = JSON.stringify(String(paper.title || "Untitled Paper"));
  const paperKey = JSON.stringify(String(paper.paper_key || "unknown-paper-key"));
  const abstract = JSON.stringify(String(paper.abstract || ""));
  return `#!/usr/bin/env node
/**
 * Auto-generated by arxiv-skill-extractor.
 * This is a runnable scaffold, not a final production implementation.
 */
const meta = {
  paperKey: ${paperKey},
  title: ${title},
};

function summarizeAbstract(input) {
  const text = String(input || "").trim();
  if (!text) return "No abstract available.";
  return text.length > 280 ? text.slice(0, 280) + "..." : text;
}

function main() {
  console.log("[arxiv-learned-skill] Running scaffold");
  console.log("Paper:", meta.paperKey);
  console.log("Title:", meta.title);
  console.log("Abstract summary:", summarizeAbstract(${abstract}));
  console.log("TODO: implement algorithm-specific code here.");
}

if (require.main === module) {
  main();
}

module.exports = { main };
`;
}

function loadPendingTask() {
  if (!fs.existsSync(PENDING_TASK_PATH)) {
    return null;
  }
  try {
    return JSON.parse(fs.readFileSync(PENDING_TASK_PATH, "utf8"));
  } catch (e) {
    throw new Error(`Invalid pending task JSON: ${e.message}`);
  }
}

function savePendingTask(task) {
  ensureDir(path.dirname(PENDING_TASK_PATH));
  fs.writeFileSync(PENDING_TASK_PATH, JSON.stringify(task, null, 2));
}

async function extractSkill(paperKey, options = {}) {
  const { getPaper } = require(path.resolve(__dirname, "../arxiv-paper-reviews/paper_client.js"));
  const finalPaperKey = String(paperKey || "").trim();
  if (!finalPaperKey) {
    throw new Error("paperKey is required");
  }

  console.log(`[arxiv-skill-extractor] Extracting skill from paper ${finalPaperKey}...`);
  const paper = options.paper || (await getPaper(finalPaperKey));

  // Use title for readable slug, fallback to paper key if title missing
  const titleSlug = sanitizeSlug(paper.title || "").slice(0, 40);
  const keySlug = sanitizeSlug(paper.paper_key || finalPaperKey).slice(0, 8);
  
  // Naming convention: arxiv-<title-slug>
  // If title is too short or missing, append key for uniqueness
  let skillName = `arxiv-${titleSlug}`;
  if (titleSlug.length < 5) {
    skillName = `arxiv-learned-${keySlug}`;
  }

  const skillDir = path.join(WORKSPACE_ROOT, "skills", skillName);
  const scriptsDir = path.join(skillDir, "scripts");

  ensureDir(scriptsDir);
  fs.writeFileSync(path.join(skillDir, "SKILL.md"), renderSkillMarkdown({ skillName, paper }));
  fs.writeFileSync(path.join(scriptsDir, "run.js"), renderNodeRunner({ paper }));
  fs.writeFileSync(path.join(skillDir, "paper.json"), `${JSON.stringify(paper, null, 2)}\n`);

  return {
    paperKey: finalPaperKey,
    status: "extracted",
    skillName,
    skillPath: skillDir,
    smokeTestCommand: `node "${path.join(scriptsDir, "run.js")}"`,
  };
}

async function main() {
  const argPaperKey = process.argv[2];
  const pendingTask = loadPendingTask();
  const paperKey = argPaperKey || (pendingTask && (pendingTask.paperKey || pendingTask.paperId));

  if (!paperKey) {
    throw new Error("No paper key provided. Usage: node index.js <paper_key>");
  }

  const result = await extractSkill(paperKey, {
    paper: pendingTask && pendingTask.paper ? pendingTask.paper : undefined,
  });

  if (pendingTask) {
    pendingTask.status = "extracted";
    pendingTask.extractedAt = new Date().toISOString();
    pendingTask.generatedSkill = {
      name: result.skillName,
      path: result.skillPath,
      smokeTestCommand: result.smokeTestCommand,
    };
    savePendingTask(pendingTask);
  }

  console.log(JSON.stringify(result, null, 2));
}

module.exports = { extractSkill, main };

if (require.main === module) {
  main().catch((error) => {
    console.error(`[arxiv-skill-extractor] Error: ${error.message}`);
    process.exit(1);
  });
}
