#!/usr/bin/env bash
# Arc Sentinel - Security Monitoring Script
# Run: bash sentinel.sh
# Exit codes: 0=ok, 1=warnings, 2=critical

set -uo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
REPORT_DIR="$SCRIPT_DIR/reports"
mkdir -p "$REPORT_DIR"

# ─── LOAD CONFIG ──────────────────────────────────────────────────────
CONF_FILE="$SCRIPT_DIR/sentinel.conf"
if [ ! -f "$CONF_FILE" ]; then
  echo "ERROR: sentinel.conf not found. Copy sentinel.conf.example to sentinel.conf and customize."
  echo "  cp $SCRIPT_DIR/sentinel.conf.example $CONF_FILE"
  exit 1
fi
# shellcheck source=/dev/null
source "$CONF_FILE"

# Validate required config
: "${DOMAINS:?Set DOMAINS in sentinel.conf}"
: "${GITHUB_USER:?Set GITHUB_USER in sentinel.conf}"
: "${MONITOR_EMAIL:?Set MONITOR_EMAIL in sentinel.conf}"
KNOWN_REPOS="${KNOWN_REPOS:-}"
HIBP_API_KEY="${HIBP_API_KEY:-}"

DATE=$(date +%Y-%m-%d)
REPORT_FILE="$REPORT_DIR/$DATE.json"
EXIT_CODE=0

# Colors
RED='\033[0;31m'
YEL='\033[0;33m'
GRN='\033[0;32m'
CYN='\033[0;36m'
RST='\033[0m'

# Tracking arrays for JSON report
declare -a ISSUES=()

log_ok()   { echo -e "  ${GRN}✓${RST} $1"; }
log_warn() { echo -e "  ${YEL}⚠${RST} $1"; ISSUES+=("{\"level\":\"warn\",\"msg\":\"$1\"}"); [ $EXIT_CODE -lt 1 ] && EXIT_CODE=1; }
log_crit() { echo -e "  ${RED}✗${RST} $1"; ISSUES+=("{\"level\":\"critical\",\"msg\":\"$1\"}"); EXIT_CODE=2; }
log_info() { echo -e "  ${CYN}ℹ${RST} $1"; }

echo ""
echo "╔══════════════════════════════════════════╗"
echo "║     Arc Sentinel - Security Scan         ║"
echo "║     $(date '+%Y-%m-%d %H:%M:%S')                  ║"
echo "╚══════════════════════════════════════════╝"
echo ""

# ─── 1. SSL CERTIFICATE CHECKS ────────────────────────────────────────
echo -e "${CYN}[1/4] SSL Certificate Expiry${RST}"

# shellcheck disable=SC2086
read -ra DOMAIN_LIST <<< $DOMAINS

for domain in "${DOMAIN_LIST[@]}"; do
  expiry=$(echo | openssl s_client -servername "$domain" -connect "$domain":443 2>/dev/null \
    | openssl x509 -noout -enddate 2>/dev/null | cut -d= -f2)

  if [ -z "$expiry" ]; then
    log_warn "$domain: Could not retrieve certificate"
    continue
  fi

  # Calculate days until expiry
  if [[ "$(uname)" == "Darwin" ]]; then
    expiry_epoch=$(date -j -f "%b %d %H:%M:%S %Y %Z" "$expiry" +%s 2>/dev/null || echo 0)
  else
    expiry_epoch=$(date -d "$expiry" +%s 2>/dev/null || echo 0)
  fi
  now_epoch=$(date +%s)
  days_left=$(( (expiry_epoch - now_epoch) / 86400 ))

  if [ "$days_left" -lt 14 ]; then
    log_crit "$domain: Certificate expires in ${days_left} days ($expiry)"
  elif [ "$days_left" -lt 30 ]; then
    log_warn "$domain: Certificate expires in ${days_left} days ($expiry)"
  else
    log_ok "$domain: Certificate valid for ${days_left} days (until $expiry)"
  fi
done

echo ""

# ─── 2. GITHUB SECURITY ───────────────────────────────────────────────
echo -e "${CYN}[2/4] GitHub Security${RST}"

if command -v gh &>/dev/null && gh auth status &>/dev/null; then
  # List repos
  repos=$(gh repo list "$GITHUB_USER" --json name,owner --limit 50 2>/dev/null)
  repo_count=$(echo "$repos" | python3 -c "import sys,json; print(len(json.load(sys.stdin)))" 2>/dev/null || echo "?")
  log_info "Found $repo_count repositories under $GITHUB_USER"

  # Check each repo for security features
  repo_names=$(echo "$repos" | python3 -c "import sys,json; [print(f'{r[\"owner\"][\"login\"]}/{r[\"name\"]}') for r in json.load(sys.stdin)]" 2>/dev/null)

  while IFS= read -r repo; do
    [ -z "$repo" ] && continue

    dep_status=$(perl -e 'alarm 10; exec @ARGV' gh api "repos/$repo/vulnerability-alerts" 2>&1 || true)
    if echo "$dep_status" | grep -qi "disabled"; then
      log_warn "$repo: Vulnerability alerts are DISABLED - consider enabling"
    elif echo "$dep_status" | grep -qi "204\|enabled"; then
      log_ok "$repo: Vulnerability alerts enabled"
    else
      log_info "$repo: Could not determine vulnerability alert status"
    fi

    if ! echo "$dep_status" | grep -qi "disabled"; then
      dep_alerts=$(gh api "repos/$repo/dependabot/alerts" --jq 'length' 2>/dev/null || echo "N/A")
      if [ "$dep_alerts" != "N/A" ] && [ "$dep_alerts" -gt 0 ] 2>/dev/null; then
        log_warn "$repo: $dep_alerts Dependabot alerts found"
      fi
    fi
  done <<< "$repo_names"

  # Recent activity check
  echo ""
  log_info "Recent GitHub activity:"
  gh api "users/$GITHUB_USER/events" --jq '.[0:5] | .[] | "    \(.type) → \(.repo.name) (\(.created_at))"' 2>/dev/null || log_warn "Could not fetch activity"

  # Check for unexpected repos
  if [ -n "$KNOWN_REPOS" ]; then
    # shellcheck disable=SC2086
    read -ra KNOWN_LIST <<< $KNOWN_REPOS
    unexpected=$(echo "$repos" | python3 -c "
import sys, json
known = set(sys.argv[1:])
repos = json.load(sys.stdin)
for r in repos:
    if r['name'] not in known:
        print(r['name'])
" "${KNOWN_LIST[@]}" 2>/dev/null)

    if [ -n "$unexpected" ]; then
      while IFS= read -r name; do
        log_warn "Unexpected repository found: $name"
      done <<< "$unexpected"
    fi
  fi

else
  log_warn "GitHub CLI not authenticated - skipping GitHub checks"
fi

echo ""

# ─── 3. BREACH MONITORING ─────────────────────────────────────────────
echo -e "${CYN}[3/4] Breach Monitoring (HaveIBeenPwned)${RST}"

if [ -n "$HIBP_API_KEY" ]; then
  hibp_response=$(curl -s -w "\n%{http_code}" \
    "https://haveibeenpwned.com/api/v3/breachedaccount/$MONITOR_EMAIL?truncateResponse=true" \
    -H "hibp-api-key: $HIBP_API_KEY" \
    -H "user-agent: ArcSentinel/1.0" 2>/dev/null)

  http_code=$(echo "$hibp_response" | tail -1)
  body=$(echo "$hibp_response" | sed '$d')

  case "$http_code" in
    200)
      breach_count=$(echo "$body" | python3 -c "import sys,json; print(len(json.load(sys.stdin)))" 2>/dev/null || echo "?")
      log_crit "$MONITOR_EMAIL found in $breach_count breaches!"
      log_info "Breaches: $body"
      ;;
    404)
      log_ok "$MONITOR_EMAIL: No breaches found"
      ;;
    429)
      log_warn "HIBP rate limited - try again later"
      ;;
    *)
      log_warn "HIBP returned HTTP $http_code"
      ;;
  esac
else
  log_info "HIBP API key not set (set HIBP_API_KEY in sentinel.conf)"
  log_info "Manual check: https://haveibeenpwned.com/ (search $MONITOR_EMAIL)"
  log_info "API key costs \$3.50/mo at https://haveibeenpwned.com/API/Key"
fi

echo ""

# ─── 4. CREDENTIAL ROTATION TRACKING ──────────────────────────────────
echo -e "${CYN}[4/4] Credential Rotation Status${RST}"

TRACKER="$SCRIPT_DIR/credential-tracker.json"

if [ -f "$TRACKER" ]; then
  python3 - "$TRACKER" << 'PYEOF'
import json, sys
from datetime import datetime

with open(sys.argv[1]) as f:
    data = json.load(f)

today = datetime.now().date()
policies = {
    "quarterly": 90,
    "6_months": 180,
    "annual": 365,
    "auto": None
}

for cred in data["credentials"]:
    name = cred["name"]
    policy = cred.get("rotation_policy", "unknown")
    last_rotated = cred.get("last_rotated")
    expires = cred.get("expires")
    auto = cred.get("auto_refreshes", False)

    if auto:
        print(f"  \033[0;32m✓\033[0m {name}: Auto-refreshes (policy: {policy})")
        continue

    # Check expiry
    if expires:
        exp_date = datetime.strptime(expires, "%Y-%m-%d").date()
        days_to_expiry = (exp_date - today).days
        if days_to_expiry < 0:
            print(f"  \033[0;31m✗\033[0m {name}: EXPIRED {abs(days_to_expiry)} days ago!")
            continue
        elif days_to_expiry < 30:
            print(f"  \033[0;33m⚠\033[0m {name}: Expires in {days_to_expiry} days ({expires})")
            continue
        elif days_to_expiry < 90:
            print(f"  \033[0;33m⚠\033[0m {name}: Expires in {days_to_expiry} days ({expires})")
            continue

    # Check rotation policy
    if last_rotated and policy in policies and policies[policy]:
        rot_date = datetime.strptime(last_rotated, "%Y-%m-%d").date()
        days_since = (today - rot_date).days
        max_days = policies[policy]
        if days_since > max_days:
            overdue = days_since - max_days
            print(f"  \033[0;33m⚠\033[0m {name}: Rotation OVERDUE by {overdue} days (last: {last_rotated}, policy: {policy})")
        elif days_since > max_days * 0.8:
            remaining = max_days - days_since
            print(f"  \033[0;33m⚠\033[0m {name}: Rotation due in ~{remaining} days (last: {last_rotated})")
        else:
            remaining = max_days - days_since
            print(f"  \033[0;32m✓\033[0m {name}: OK - {remaining} days until rotation due (last: {last_rotated})")
    elif expires:
        exp_date = datetime.strptime(expires, "%Y-%m-%d").date()
        days_to_expiry = (exp_date - today).days
        print(f"  \033[0;32m✓\033[0m {name}: {days_to_expiry} days until expiry ({expires})")
    else:
        print(f"  \033[0;36mℹ\033[0m {name}: No expiry/rotation policy defined")
PYEOF
else
  log_warn "Credential tracker file not found at $TRACKER"
fi

echo ""

# ─── SUMMARY ──────────────────────────────────────────────────────────
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
if [ $EXIT_CODE -eq 0 ]; then
  echo -e "${GRN}Status: ALL CLEAR${RST} - No issues found"
elif [ $EXIT_CODE -eq 1 ]; then
  echo -e "${YEL}Status: WARNINGS${RST} - Review items above"
else
  echo -e "${RED}Status: CRITICAL${RST} - Immediate attention needed"
fi
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"

# Save JSON report
python3 -c "
import json
from datetime import datetime
report = {
    'date': datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%SZ'),
    'exit_code': $EXIT_CODE,
    'status': ['ok','warnings','critical'][$EXIT_CODE],
    'issues': []
}
print(json.dumps(report, indent=2))
" > "$REPORT_FILE" 2>/dev/null || true

echo ""
echo "Report saved to: $REPORT_FILE"

exit $EXIT_CODE
