#!/usr/bin/env python3\n\"\"\"\nCheck Vercel AI Gateway balance via API.\n\nRequires: VERCEL_AI_GATEWAY_KEY environment variable\n\nUsage:\n    VERCEL_AI_GATEWAY_KEY=... python3 check_vercel.py\n    VERCEL_AI_GATEWAY_KEY=... python3 check_vercel.py --update\n\"\"\"\n\nimport os\nimport json\nimport sys\nfrom datetime import datetime\n\ntry:\n    import requests\nexcept ImportError:\n    print(\"❌ requests library not installed. Install with: pip install requests\")\n    sys.exit(1)\n\ndef load_config():\n    \"\"\"Load config from the skill directory\"\"\"\n    script_dir = os.path.dirname(os.path.abspath(__file__))\n    config_path = os.path.join(script_dir, '..', 'config.json')\n    \n    try:\n        with open(config_path, 'r') as f:\n            return json.load(f)\n    except FileNotFoundError:\n        return {'providers': {}, 'thresholds': {'warning': 50, 'critical': 25}}\n\ndef save_config(config):\n    \"\"\"Save config back to file\"\"\"\n    script_dir = os.path.dirname(os.path.abspath(__file__))\n    config_path = os.path.join(script_dir, '..', 'config.json')\n    \n    with open(config_path, 'w') as f:\n        json.dump(config, f, indent=2)\n\ndef check_vercel_balance(api_key):\n    \"\"\"\n    Check Vercel AI Gateway balance using API.\n    \n    Returns: (balance, limit, timestamp)\n    \"\"\"\n    try:\n        response = requests.get(\n            'https://api.vercel.com/v1/billing',\n            headers={\n                'Authorization': f'Bearer {api_key}',\n                'Content-Type': 'application/json'\n            }\n        )\n        \n        if response.status_code != 200:\n            print(f\"❌ API error: {response.status_code}\")\n            if response.status_code == 401:\n                print(\"   Invalid API key\")\n            return None, None, None\n        \n        data = response.json()\n        \n        # Vercel returns balance and usage info\n        balance = float(data.get('balance', 0))\n        limit = float(data.get('limit', balance * 2))\n        \n        return balance, limit, datetime.now().isoformat()\n    \n    except Exception as e:\n        print(f\"❌ Failed to check Vercel balance: {str(e)}\")\n        return None, None, None\n\ndef check_and_display(api_key, update=False):\n    \"\"\"\n    Check balance and optionally update config.\n    \"\"\"\n    print(\"🔍 Checking Vercel AI Gateway balance...\")\n    \n    balance, limit, timestamp = check_vercel_balance(api_key)\n    \n    if balance is None:\n        return 1\n    \n    print(f\"✅ Vercel balance: ${balance:.2f}\")\n    print(f\"   Limit: ${limit:.2f}\")\n    print(f\"   Checked: {timestamp}\")\n    \n    if update:\n        config = load_config()\n        if 'vercel' not in config.get('providers', {}):\n            config.setdefault('providers', {})['vercel'] = {\n                'enabled': True,\n                'max_credits': limit,\n                'current_credits': balance,\n                'api_key_env': 'VERCEL_AI_GATEWAY_KEY',\n                'last_sync': timestamp,\n                'notes': 'Auto-tracked via Vercel API'\n            }\n        else:\n            config['providers']['vercel']['current_credits'] = balance\n            config['providers']['vercel']['max_credits'] = limit\n            config['providers']['vercel']['last_sync'] = timestamp\n        \n        save_config(config)\n        print(\"   ✅ Config updated\")\n    \n    return 0\n\nif __name__ == \"__main__\":\n    api_key = os.getenv('VERCEL_AI_GATEWAY_KEY')\n    \n    if not api_key:\n        print(\"❌ VERCEL_AI_GATEWAY_KEY environment variable not set\")\n        print(\"\\nUsage:\")\n        print(\"  VERCEL_AI_GATEWAY_KEY=... python3 check_vercel.py\")\n        print(\"  VERCEL_AI_GATEWAY_KEY=... python3 check_vercel.py --update\")\n        sys.exit(1)\n    \n    update = '--update' in sys.argv\n    \n    sys.exit(check_and_display(api_key, update=update))\n"
