#!/usr/bin/env python3\n\"\"\"\nCheck OpenAI credits balance via API.\n\nRequires: OPENAI_API_KEY environment variable (org admin key, not user key)\n\nUsage:\n    OPENAI_API_KEY=sk-... python3 check_openai.py\n    OPENAI_API_KEY=sk-... python3 check_openai.py --update\n\"\"\"\n\nimport os\nimport json\nimport sys\nfrom datetime import datetime\n\ntry:\n    from openai import OpenAI\nexcept ImportError:\n    print(\"❌ OpenAI library not installed. Install with: pip install openai\")\n    sys.exit(1)\n\ndef load_config():\n    \"\"\"Load config from the skill directory\"\"\"\n    script_dir = os.path.dirname(os.path.abspath(__file__))\n    config_path = os.path.join(script_dir, '..', 'config.json')\n    \n    try:\n        with open(config_path, 'r') as f:\n            return json.load(f)\n    except FileNotFoundError:\n        return {'providers': {}, 'thresholds': {'warning': 50, 'critical': 25}}\n\ndef save_config(config):\n    \"\"\"Save config back to file\"\"\"\n    script_dir = os.path.dirname(os.path.abspath(__file__))\n    config_path = os.path.join(script_dir, '..', 'config.json')\n    \n    with open(config_path, 'w') as f:\n        json.dump(config, f, indent=2)\n\ndef check_openai_balance(api_key):\n    \"\"\"\n    Check OpenAI balance using API.\n    \n    Requires: Organization admin API key (not user key)\n    Returns: (total_balance, date)\n    \"\"\"\n    try:\n        client = OpenAI(api_key=api_key, organization=None)\n        \n        # Get billing info\n        billing_info = client.beta.billing.credit_grants.list()\n        \n        total_balance = 0\n        for grant in billing_info.data:\n            if hasattr(grant, 'balance'):\n                total_balance += float(grant.balance)\n        \n        return total_balance, datetime.now().isoformat()\n    \n    except Exception as e:\n        print(f\"❌ Failed to check OpenAI balance: {str(e)}\")\n        print(\"\\n💡 Note: This requires an organization admin API key, not a user key.\")\n        print(\"   Get it from: https://platform.openai.com/account/api-keys (select org)\")\n        return None, None\n\ndef check_and_display(api_key, update=False):\n    \"\"\"\n    Check balance and optionally update config.\n    \"\"\"\n    print(\"🔍 Checking OpenAI balance...\")\n    \n    balance, timestamp = check_openai_balance(api_key)\n    \n    if balance is None:\n        return 1\n    \n    print(f\"✅ OpenAI balance: ${balance:.2f}\")\n    print(f\"   Checked: {timestamp}\")\n    \n    if update:\n        config = load_config()\n        if 'openai' not in config.get('providers', {}):\n            config.setdefault('providers', {})['openai'] = {\n                'enabled': True,\n                'max_credits': balance,\n                'current_credits': balance,\n                'api_key_env': 'OPENAI_API_KEY',\n                'notes': 'Auto-tracked via OpenAI API'\n            }\n        else:\n            config['providers']['openai']['current_credits'] = balance\n            config['providers']['openai']['last_sync'] = timestamp\n        \n        save_config(config)\n        print(\"   ✅ Config updated\")\n    \n    return 0\n\nif __name__ == \"__main__\":\n    api_key = os.getenv('OPENAI_API_KEY')\n    \n    if not api_key:\n        print(\"❌ OPENAI_API_KEY environment variable not set\")\n        print(\"\\nUsage:\")\n        print(\"  OPENAI_API_KEY=sk-... python3 check_openai.py\")\n        print(\"  OPENAI_API_KEY=sk-... python3 check_openai.py --update\")\n        sys.exit(1)\n    \n    update = '--update' in sys.argv\n    \n    sys.exit(check_and_display(api_key, update=update))\n"
