#!/usr/bin/env python3\n\"\"\"\nCheck all available API providers at once.\n\nRuns check_openai.py, check_openrouter.py, and check_vercel.py\nand displays a summary.\n\nUsage:\n    python3 check_all_apis.py\n    python3 check_all_apis.py --update  (updates config)\n\"\"\"\n\nimport os\nimport subprocess\nimport sys\nimport json\nfrom render_healthbar import render_bar\n\nAPIS = [\n    {\n        'name': 'OpenAI',\n        'script': 'check_openai.py',\n        'env_key': 'OPENAI_API_KEY'\n    },\n    {\n        'name': 'OpenRouter',\n        'script': 'check_openrouter.py',\n        'env_key': 'OPENROUTER_API_KEY'\n    },\n    {\n        'name': 'Vercel',\n        'script': 'check_vercel.py',\n        'env_key': 'VERCEL_AI_GATEWAY_KEY'\n    }\n]\n\ndef load_config():\n    \"\"\"Load current config\"\"\"\n    script_dir = os.path.dirname(os.path.abspath(__file__))\n    config_path = os.path.join(script_dir, '..', 'config.json')\n    \n    try:\n        with open(config_path, 'r') as f:\n            return json.load(f)\n    except FileNotFoundError:\n        return {'providers': {}}\n\ndef check_all_apis(update=False):\n    \"\"\"\n    Run all API checks and display results.\n    \"\"\"\n    script_dir = os.path.dirname(os.path.abspath(__file__))\n    \n    print(\"🔍 Checking all API providers...\\n\")\n    \n    results = []\n    failed = 0\n    \n    for api in APIS:\n        if not os.getenv(api['env_key']):\n            print(f\"⏭️  {api['name']}: Skipped (${api['env_key']} not set)\")\n            continue\n        \n        cmd = ['python3', os.path.join(script_dir, api['script'])]\n        if update:\n            cmd.append('--update')\n        \n        try:\n            result = subprocess.run(cmd, capture_output=True, text=True, timeout=10)\n            \n            if result.returncode == 0:\n                print(f\"✅ {api['name']}: {result.stdout.strip()}\")\n                results.append(api['name'])\n            else:\n                print(f\"❌ {api['name']}: Failed\")\n                if result.stderr:\n                    print(f\"   {result.stderr}\")\n                failed += 1\n        \n        except subprocess.TimeoutExpired:\n            print(f\"⏱️  {api['name']}: Timeout\")\n            failed += 1\n        except Exception as e:\n            print(f\"❌ {api['name']}: {str(e)}\")\n            failed += 1\n    \n    print()\n    print(f\"Summary: {len(results)}/{len(APIS)} providers checked successfully\")\n    \n    if update:\n        print(\"✅ Config updated with latest balances\")\n        print(\"\\nRun: python3 show_credits.py\")\n    \n    return 0 if failed == 0 else 1\n\nif __name__ == \"__main__\":\n    update = '--update' in sys.argv\n    sys.exit(check_all_apis(update=update))\n"
