#!/usr/bin/env python3\n\"\"\"\nCheck OpenRouter credits balance via API.\n\nRequires: OPENROUTER_API_KEY environment variable\n\nUsage:\n    OPENROUTER_API_KEY=sk-... python3 check_openrouter.py\n    OPENROUTER_API_KEY=sk-... python3 check_openrouter.py --update\n\"\"\"\n\nimport os\nimport json\nimport sys\nfrom datetime import datetime\n\ntry:\n    import requests\nexcept ImportError:\n    print(\"❌ requests library not installed. Install with: pip install requests\")\n    sys.exit(1)\n\ndef load_config():\n    \"\"\"Load config from the skill directory\"\"\"\n    script_dir = os.path.dirname(os.path.abspath(__file__))\n    config_path = os.path.join(script_dir, '..', 'config.json')\n    \n    try:\n        with open(config_path, 'r') as f:\n            return json.load(f)\n    except FileNotFoundError:\n        return {'providers': {}, 'thresholds': {'warning': 50, 'critical': 25}}\n\ndef save_config(config):\n    \"\"\"Save config back to file\"\"\"\n    script_dir = os.path.dirname(os.path.abspath(__file__))\n    config_path = os.path.join(script_dir, '..', 'config.json')\n    \n    with open(config_path, 'w') as f:\n        json.dump(config, f, indent=2)\n\ndef check_openrouter_balance(api_key):\n    \"\"\"\n    Check OpenRouter credits using API.\n    \n    Returns: (total_credits, limit, timestamp)\n    \"\"\"\n    try:\n        response = requests.get(\n            'https://openrouter.ai/api/v1/auth/key',\n            headers={\n                'Authorization': f'Bearer {api_key}',\n                'Content-Type': 'application/json'\n            }\n        )\n        \n        if response.status_code != 200:\n            print(f\"❌ API error: {response.status_code}\")\n            if response.status_code == 401:\n                print(\"   Invalid API key\")\n            return None, None, None\n        \n        data = response.json()\n        \n        # OpenRouter returns credits and limit\n        credits = float(data.get('balance', data.get('credits', 0)))\n        limit = float(data.get('limit', credits * 2))  # Estimate if not provided\n        \n        return credits, limit, datetime.now().isoformat()\n    \n    except Exception as e:\n        print(f\"❌ Failed to check OpenRouter balance: {str(e)}\")\n        return None, None, None\n\ndef check_and_display(api_key, update=False):\n    \"\"\"\n    Check balance and optionally update config.\n    \"\"\"\n    print(\"🔍 Checking OpenRouter balance...\")\n    \n    credits, limit, timestamp = check_openrouter_balance(api_key)\n    \n    if credits is None:\n        return 1\n    \n    print(f\"✅ OpenRouter balance: ${credits:.2f}\")\n    print(f\"   Limit: ${limit:.2f}\")\n    print(f\"   Checked: {timestamp}\")\n    \n    if update:\n        config = load_config()\n        if 'openrouter' not in config.get('providers', {}):\n            config.setdefault('providers', {})['openrouter'] = {\n                'enabled': True,\n                'max_credits': limit,\n                'current_credits': credits,\n                'api_key_env': 'OPENROUTER_API_KEY',\n                'last_sync': timestamp,\n                'notes': 'Auto-tracked via OpenRouter API'\n            }\n        else:\n            config['providers']['openrouter']['current_credits'] = credits\n            config['providers']['openrouter']['max_credits'] = limit\n            config['providers']['openrouter']['last_sync'] = timestamp\n        \n        save_config(config)\n        print(\"   ✅ Config updated\")\n    \n    return 0\n\nif __name__ == \"__main__\":\n    api_key = os.getenv('OPENROUTER_API_KEY')\n    \n    if not api_key:\n        print(\"❌ OPENROUTER_API_KEY environment variable not set\")\n        print(\"\\nUsage:\")\n        print(\"  OPENROUTER_API_KEY=sk-... python3 check_openrouter.py\")\n        print(\"  OPENROUTER_API_KEY=sk-... python3 check_openrouter.py --update\")\n        sys.exit(1)\n    \n    update = '--update' in sys.argv\n    \n    sys.exit(check_and_display(api_key, update=update))\n"
