// AnyCrawl Skill for OpenClaw
// API Docs: https://docs.anycrawl.dev
// API Base: https://api.anycrawl.dev/v1

const API_BASE = "https://api.anycrawl.dev/v1";
const API_KEY = process.env.ANYCRAWL_API_KEY;

async function anycrawlRequest(endpoint, options = {}) {
  if (!API_KEY) {
    throw new Error(
      "AnyCrawl API Key not found!\n\n" +
      "Please set your API key:\n" +
      "export ANYCRAWL_API_KEY=\"your-key\"\n\n" +
      "Get your API key at: https://anycrawl.dev"
    );
  }
  
  const url = `${API_BASE}${endpoint}`;
  const headers = {
    "Authorization": `Bearer ${API_KEY}`,
    "Content-Type": "application/json",
    ...options.headers
  };
  
  const response = await fetch(url, {
    ...options,
    headers
  });
  
  const data = await response.json();
  
  if (!response.ok) {
    throw new Error(`AnyCrawl API Error: ${data.error || data.message || response.statusText}`);
  }
  
  return data;
}

/**
 * Scrape a single URL and convert to LLM-ready structured data
 * 
 * @param {Object} params
 * @param {string} params.url - URL to scrape (required)
 * @param {string} params.engine - Scraping engine: 'cheerio' (default), 'playwright', 'puppeteer'
 * @param {Array} params.formats - Output formats: ['markdown'], ['html'], ['text'], ['json'], ['screenshot'], etc.
 * @param {number} params.timeout - Timeout in milliseconds (default: 30000)
 * @param {number} params.wait_for - Delay before extraction in ms (browser engines only)
 * @param {string|Object|Array} params.wait_for_selector - Wait for CSS selectors
 * @param {Array} params.include_tags - Include only these HTML tags
 * @param {Array} params.exclude_tags - Exclude these HTML tags
 * @param {string} params.proxy - Proxy URL (http://proxy:port)
 * @param {Object} params.json_options - JSON extraction options with schema and prompt
 * @param {string} params.extract_source - Source for JSON extraction: 'markdown' (default) or 'html'
 * 
 * @returns {Promise<Object>} Scraped content in requested formats
 */
export async function anycrawl_scrape({ 
  url, 
  engine = "cheerio",
  formats = ["markdown"],
  timeout,
  wait_for,
  wait_for_selector,
  include_tags,
  exclude_tags,
  proxy,
  json_options,
  extract_source
}) {
  if (!url) {
    throw new Error("URL is required");
  }
  
  const body = {
    url,
    engine,
    formats
  };
  
  if (timeout !== undefined) body.timeout = timeout;
  if (wait_for !== undefined) body.wait_for = wait_for;
  if (wait_for_selector !== undefined) body.wait_for_selector = wait_for_selector;
  if (include_tags !== undefined) body.include_tags = include_tags;
  if (exclude_tags !== undefined) body.exclude_tags = exclude_tags;
  if (proxy !== undefined) body.proxy = proxy;
  if (json_options !== undefined) body.json_options = json_options;
  if (extract_source !== undefined) body.extract_source = extract_source;
  
  return await anycrawlRequest("/scrape", {
    method: "POST",
    body: JSON.stringify(body)
  });
}

/**
 * Search Google and return structured results
 * 
 * @param {Object} params
 * @param {string} params.query - Search query string (required)
 * @param {string} params.engine - Search engine: 'google' (default)
 * @param {number} params.limit - Max results per page (default: 10)
 * @param {number} params.offset - Number of results to skip (default: 0)
 * @param {number} params.pages - Number of pages to retrieve (default: 1, max: 20)
 * @param {string} params.lang - Language locale (e.g., 'en', 'zh', 'vi', default: 'en')
 * @param {number} params.safe_search - Safe search: 0 (off), 1 (medium), 2 (high)
 * @param {Object} params.scrape_options - Optional: scrape each result URL with these options
 * 
 * @returns {Promise<Object>} Search results and suggestions
 */
export async function anycrawl_search({
  query,
  engine = "google",
  limit,
  offset,
  pages,
  lang,
  safe_search,
  scrape_options
}) {
  if (!query) {
    throw new Error("Query is required");
  }
  
  const body = { query };
  
  if (engine !== undefined) body.engine = engine;
  if (limit !== undefined) body.limit = limit;
  if (offset !== undefined) body.offset = offset;
  if (pages !== undefined) body.pages = pages;
  if (lang !== undefined) body.lang = lang;
  if (safe_search !== undefined) body.safe_search = safe_search;
  if (scrape_options !== undefined) body.scrape_options = scrape_options;
  
  return await anycrawlRequest("/search", {
    method: "POST",
    body: JSON.stringify(body)
  });
}

/**
 * Start a crawl job for an entire website
 * 
 * @param {Object} params
 * @param {string} params.url - Seed URL to start crawling (required)
 * @param {string} params.engine - Scraping engine: 'cheerio' (default), 'playwright', 'puppeteer'
 * @param {string} params.strategy - Crawl scope: 'all', 'same-domain' (default), 'same-hostname', 'same-origin'
 * @param {number} params.max_depth - Max depth from seed URL (default: 10)
 * @param {number} params.limit - Max pages to crawl (default: 100)
 * @param {Array} params.include_paths - Path patterns to include (glob-like)
 * @param {Array} params.exclude_paths - Path patterns to exclude (glob-like)
 * @param {Array} params.scrape_paths - Only scrape URLs matching these patterns
 * @param {Object} params.scrape_options - Per-page scrape options
 * 
 * @returns {Promise<Object>} Job ID and status
 */
export async function anycrawl_crawl_start({
  url,
  engine = "cheerio",
  strategy = "same-domain",
  max_depth,
  limit,
  include_paths,
  exclude_paths,
  scrape_paths,
  scrape_options
}) {
  if (!url) {
    throw new Error("URL is required");
  }
  
  const body = {
    url,
    engine,
    strategy
  };
  
  if (max_depth !== undefined) body.max_depth = max_depth;
  if (limit !== undefined) body.limit = limit;
  if (include_paths !== undefined) body.include_paths = include_paths;
  if (exclude_paths !== undefined) body.exclude_paths = exclude_paths;
  if (scrape_paths !== undefined) body.scrape_paths = scrape_paths;
  if (scrape_options !== undefined) body.scrape_options = scrape_options;
  
  return await anycrawlRequest("/crawl", {
    method: "POST",
    body: JSON.stringify(body)
  });
}

/**
 * Get crawl job status
 * 
 * @param {Object} params
 * @param {string} params.job_id - Crawl job ID (required)
 * 
 * @returns {Promise<Object>} Job status, progress, and stats
 */
export async function anycrawl_crawl_status({ job_id }) {
  if (!job_id) {
    throw new Error("Job ID is required");
  }
  
  return await anycrawlRequest(`/crawl/${job_id}/status`);
}

/**
 * Get crawl results (paginated)
 * 
 * @param {Object} params
 * @param {string} params.job_id - Crawl job ID (required)
 * @param {number} params.skip - Number of results to skip (default: 0)
 * 
 * @returns {Promise<Object>} Crawled pages with content
 */
export async function anycrawl_crawl_results({ job_id, skip = 0 }) {
  if (!job_id) {
    throw new Error("Job ID is required");
  }
  
  return await anycrawlRequest(`/crawl/${job_id}?skip=${skip}`);
}

/**
 * Cancel a crawl job
 * 
 * @param {Object} params
 * @param {string} params.job_id - Crawl job ID (required)
 * 
 * @returns {Promise<Object>} Cancellation confirmation
 */
export async function anycrawl_crawl_cancel({ job_id }) {
  if (!job_id) {
    throw new Error("Job ID is required");
  }
  
  return await anycrawlRequest(`/crawl/${job_id}`, {
    method: "DELETE"
  });
}

/**
 * Quick search and scrape - Search Google then scrape top results
 * 
 * @param {Object} params
 * @param {string} params.query - Search query (required)
 * @param {number} params.max_results - Max results to scrape (default: 3)
 * @param {string} params.scrape_engine - Engine for scraping: 'cheerio' (default) or 'playwright'
 * @param {Array} params.formats - Output formats for scraped content
 * @param {string} params.lang - Search language
 * 
 * @returns {Promise<Object>} Search results with scraped content
 */
export async function anycrawl_search_and_scrape({
  query,
  max_results = 3,
  scrape_engine = "cheerio",
  formats = ["markdown"],
  lang
}) {
  if (!query) {
    throw new Error("Query is required");
  }
  
  // First, search
  const searchBody = { query, limit: max_results };
  if (lang) searchBody.lang = lang;
  
  const searchResults = await anycrawlRequest("/search", {
    method: "POST",
    body: JSON.stringify(searchBody)
  });
  
  if (!searchResults.success || !searchResults.data) {
    return searchResults;
  }
  
  // Filter only web results (not suggestions) - source can be "AC-Engine" or "Google Search Result"
  const webResults = searchResults.data.filter(r => r.url && r.category === "web");
  
  // Scrape top results
  const scrapedResults = [];
  for (const result of webResults.slice(0, max_results)) {
    try {
      const scrapeResult = await anycrawlRequest("/scrape", {
        method: "POST",
        body: JSON.stringify({
          url: result.url,
          engine: scrape_engine,
          formats
        })
      });
      
      scrapedResults.push({
        ...result,
        scraped: scrapeResult.success ? scrapeResult.data : null,
        scrape_error: scrapeResult.success ? null : scrapeResult.error
      });
    } catch (error) {
      scrapedResults.push({
        ...result,
        scraped: null,
        scrape_error: error.message
      });
    }
  }
  
  return {
    success: true,
    query,
    search_results: searchResults.data,
    scraped_results: scrapedResults
  };
}

// Export all functions for OpenClaw
export default {
  anycrawl_scrape,
  anycrawl_search,
  anycrawl_crawl_start,
  anycrawl_crawl_status,
  anycrawl_crawl_results,
  anycrawl_crawl_cancel,
  anycrawl_search_and_scrape
};
