# Ansible Modules Cheatsheet

## Package Management

### apt (Debian/Ubuntu)
```yaml
# Install package
- ansible.builtin.apt:
    name: nginx
    state: present

# Install multiple packages
- ansible.builtin.apt:
    name:
      - nginx
      - curl
      - git
    state: present

# Install specific version
- ansible.builtin.apt:
    name: nginx=1.18.0-0ubuntu1
    state: present

# Remove package
- ansible.builtin.apt:
    name: nginx
    state: absent

# Update cache
- ansible.builtin.apt:
    update_cache: yes
    cache_valid_time: 3600

# Upgrade all packages
- ansible.builtin.apt:
    upgrade: safe  # or: full, dist
```

### yum/dnf (RHEL/CentOS)
```yaml
- ansible.builtin.yum:
    name: nginx
    state: present

- ansible.builtin.dnf:
    name: nginx
    state: present
```

## Files & Directories

### file
```yaml
# Create directory
- ansible.builtin.file:
    path: /opt/myapp
    state: directory
    owner: www-data
    group: www-data
    mode: '0755'

# Create symlink
- ansible.builtin.file:
    src: /opt/myapp/current
    dest: /var/www/html
    state: link

# Delete file/directory
- ansible.builtin.file:
    path: /tmp/garbage
    state: absent

# Touch file
- ansible.builtin.file:
    path: /opt/myapp/.deployed
    state: touch
    mode: '0644'
```

### copy
```yaml
# Copy file
- ansible.builtin.copy:
    src: files/nginx.conf
    dest: /etc/nginx/nginx.conf
    owner: root
    group: root
    mode: '0644'
    backup: yes

# Copy content directly
- ansible.builtin.copy:
    content: |
      server {
        listen 80;
      }
    dest: /etc/nginx/conf.d/default.conf
```

### template
```yaml
- ansible.builtin.template:
    src: nginx.conf.j2
    dest: /etc/nginx/nginx.conf
    owner: root
    mode: '0644'
    validate: 'nginx -t -c %s'
```

### lineinfile
```yaml
# Ensure line exists
- ansible.builtin.lineinfile:
    path: /etc/ssh/sshd_config
    regexp: '^PermitRootLogin'
    line: 'PermitRootLogin no'

# Add line if not exists
- ansible.builtin.lineinfile:
    path: /etc/hosts
    line: '192.168.1.100 myserver'
    state: present

# Remove line
- ansible.builtin.lineinfile:
    path: /etc/crontab
    regexp: '^.*/opt/oldscript.sh.*$'
    state: absent
```

### blockinfile
```yaml
- ansible.builtin.blockinfile:
    path: /etc/ssh/sshd_config
    block: |
      Match User deploy
        PasswordAuthentication no
        PubkeyAuthentication yes
    marker: "# {mark} ANSIBLE MANAGED - deploy user"
```

## Users & Groups

### user
```yaml
# Create user
- ansible.builtin.user:
    name: deploy
    groups: [sudo, docker]
    shell: /bin/bash
    create_home: yes

# Remove user
- ansible.builtin.user:
    name: olduser
    state: absent
    remove: yes  # Remove home directory
```

### group
```yaml
- ansible.builtin.group:
    name: developers
    state: present
```

### authorized_key
```yaml
- ansible.posix.authorized_key:
    user: deploy
    key: "{{ lookup('file', '~/.ssh/id_ed25519.pub') }}"
    state: present
    exclusive: no  # Don't remove other keys
```

## Services

### systemd
```yaml
# Start and enable service
- ansible.builtin.systemd:
    name: nginx
    state: started
    enabled: yes

# Restart service
- ansible.builtin.systemd:
    name: nginx
    state: restarted

# Reload systemd daemon
- ansible.builtin.systemd:
    daemon_reload: yes
```

### service (generic)
```yaml
- ansible.builtin.service:
    name: nginx
    state: started
    enabled: yes
```

## Commands

### command
```yaml
# Simple command
- ansible.builtin.command: /opt/script.sh

# With arguments
- ansible.builtin.command:
    cmd: /opt/script.sh --option value
    chdir: /opt
    creates: /opt/.done  # Skip if file exists
```

### shell
```yaml
# Pipe and redirects
- ansible.builtin.shell: cat /etc/passwd | grep deploy > /tmp/user.txt

# With environment
- ansible.builtin.shell: echo $MY_VAR
  environment:
    MY_VAR: "hello"
```

### script
```yaml
# Run local script on remote
- ansible.builtin.script: scripts/setup.sh
  args:
    creates: /opt/.setup-done
```

## Git

```yaml
- ansible.builtin.git:
    repo: https://github.com/user/repo.git
    dest: /opt/myapp
    version: main  # branch, tag, or commit
    force: yes     # Discard local changes
```

## Archive

### unarchive
```yaml
- ansible.builtin.unarchive:
    src: files/app.tar.gz
    dest: /opt/myapp
    remote_src: no  # src is on control machine

- ansible.builtin.unarchive:
    src: /tmp/app.tar.gz
    dest: /opt/myapp
    remote_src: yes  # src is on remote
```

### archive
```yaml
- community.general.archive:
    path: /opt/myapp
    dest: /tmp/backup.tar.gz
    format: gz
```

## Networking

### ufw (Ubuntu firewall)
```yaml
- community.general.ufw:
    rule: allow
    port: '22'
    proto: tcp

- community.general.ufw:
    state: enabled
    policy: deny
    direction: incoming
```

### wait_for
```yaml
# Wait for port to be open
- ansible.builtin.wait_for:
    port: 80
    host: localhost
    timeout: 60

# Wait for file
- ansible.builtin.wait_for:
    path: /tmp/ready.txt
    state: present
```

## Debug & Info

### debug
```yaml
- ansible.builtin.debug:
    msg: "Variable value: {{ my_var }}"

- ansible.builtin.debug:
    var: ansible_facts
```

### assert
```yaml
- ansible.builtin.assert:
    that:
      - my_var is defined
      - my_var | int > 0
    fail_msg: "my_var must be a positive integer"
```

### stat
```yaml
- ansible.builtin.stat:
    path: /etc/nginx/nginx.conf
  register: nginx_conf

- ansible.builtin.debug:
    msg: "File exists"
  when: nginx_conf.stat.exists
```

## Conditionals

```yaml
# When
- ansible.builtin.apt:
    name: nginx
  when: ansible_os_family == "Debian"

# Multiple conditions
- ansible.builtin.debug:
    msg: "Production Debian"
  when:
    - ansible_os_family == "Debian"
    - env == "production"

# Or conditions
- ansible.builtin.debug:
    msg: "Debian or Ubuntu"
  when: ansible_distribution == "Debian" or ansible_distribution == "Ubuntu"
```

## Loops

```yaml
# Simple loop
- ansible.builtin.apt:
    name: "{{ item }}"
  loop:
    - nginx
    - curl
    - git

# Dict loop
- ansible.builtin.user:
    name: "{{ item.name }}"
    groups: "{{ item.groups }}"
  loop:
    - { name: 'alice', groups: 'sudo' }
    - { name: 'bob', groups: 'developers' }

# With index
- ansible.builtin.debug:
    msg: "{{ index }}: {{ item }}"
  loop:
    - a
    - b
    - c
  loop_control:
    index_var: index
```

## Registers & Results

```yaml
- ansible.builtin.command: whoami
  register: result

- ansible.builtin.debug:
    msg: "Running as {{ result.stdout }}"

# Common result attributes
# result.stdout - command output
# result.stderr - error output
# result.rc - return code
# result.changed - whether task changed anything
# result.failed - whether task failed
```
