# Ansible Best Practices

## Directory Layout

```
production/                # Production inventory
staging/                   # Staging inventory
group_vars/
   all.yml                 # Variables for all groups
   webservers.yml          # Variables for webservers group
host_vars/
   hostname1.yml           # Variables for specific host
roles/
   common/                 # Role: common tasks
   webservers/             # Role: webserver setup
site.yml                   # Master playbook
webservers.yml             # Webserver playbook
```

## Naming Conventions

### Files
- Use `.yml` extension (not `.yaml`)
- Lowercase with hyphens: `web-servers.yml`
- Playbooks: descriptive action names: `deploy-app.yml`, `setup-database.yml`

### Variables
- Prefix role variables with role name: `nginx_port`, `nodejs_version`
- Use snake_case: `deploy_user`, `ssh_port`
- Boolean variables: use `_enabled` suffix: `firewall_enabled`

### Tasks
- Always use `name:` - descriptive, starts with verb
- Good: `name: Install nginx web server`
- Bad: `name: nginx` or no name at all

## Idempotency

Every task should be safe to run multiple times:

```yaml
# ✅ Good - idempotent
- name: Ensure nginx is installed
  ansible.builtin.apt:
    name: nginx
    state: present

# ❌ Bad - not idempotent
- name: Install nginx
  ansible.builtin.shell: apt-get install nginx
```

## Use Handlers

Don't restart services inline:

```yaml
# ✅ Good - use handlers
- name: Update SSH config
  ansible.builtin.template:
    src: sshd_config.j2
    dest: /etc/ssh/sshd_config
  notify: Restart SSH

# handlers/main.yml
- name: Restart SSH
  ansible.builtin.systemd:
    name: sshd
    state: restarted

# ❌ Bad - inline restart
- name: Update SSH config
  ansible.builtin.template:
    src: sshd_config.j2
    dest: /etc/ssh/sshd_config

- name: Restart SSH
  ansible.builtin.systemd:
    name: sshd
    state: restarted
```

## Use FQCN

Always use Fully Qualified Collection Names:

```yaml
# ✅ Good
- ansible.builtin.apt:
    name: nginx

# ⚠️ Acceptable but less clear
- apt:
    name: nginx
```

## Explicit State

Always specify state:

```yaml
# ✅ Good
- ansible.builtin.apt:
    name: nginx
    state: present

# ⚠️ Implicit (depends on module default)
- ansible.builtin.apt:
    name: nginx
```

## Variables

### Priority (highest to lowest)
1. Extra vars (`-e "var=value"`)
2. Task vars
3. Block vars
4. Role vars
5. Play vars
6. Host vars
7. Group vars
8. Role defaults

### Secrets with Vault

```bash
# Create encrypted file
ansible-vault create group_vars/all/vault.yml

# Structure: prefix vault variables
vault_db_password: "secret123"
vault_api_key: "key123"

# Reference in vars file
db_password: "{{ vault_db_password }}"
api_key: "{{ vault_api_key }}"
```

## Tags

Use tags for selective execution:

```yaml
- name: Install packages
  ansible.builtin.apt:
    name: nginx
  tags: [packages, nginx]

- name: Configure nginx
  ansible.builtin.template:
    src: nginx.conf.j2
    dest: /etc/nginx/nginx.conf
  tags: [config, nginx]
```

```bash
# Run only tagged tasks
ansible-playbook site.yml --tags "config"

# Skip tagged tasks
ansible-playbook site.yml --skip-tags "packages"
```

## Error Handling

### Ignore Errors (carefully)
```yaml
- name: Check if service exists
  ansible.builtin.command: systemctl status myservice
  register: service_status
  failed_when: false
  changed_when: false
```

### Block/Rescue/Always
```yaml
- name: Handle potential failure
  block:
    - name: Try risky operation
      ansible.builtin.command: /opt/risky-script.sh
  rescue:
    - name: Handle failure
      ansible.builtin.debug:
        msg: "Operation failed, cleaning up..."
  always:
    - name: Always run this
      ansible.builtin.debug:
        msg: "Cleanup complete"
```

## Performance

### Gather Facts Selectively
```yaml
- hosts: all
  gather_facts: no  # Skip if not needed
  
- hosts: all
  gather_subset:
    - network
    - hardware
```

### Async for Long Tasks
```yaml
- name: Long running task
  ansible.builtin.command: /opt/long-script.sh
  async: 3600  # Max runtime
  poll: 0      # Don't wait
  register: long_task

- name: Check on long task
  ansible.builtin.async_status:
    jid: "{{ long_task.ansible_job_id }}"
  register: job_result
  until: job_result.finished
  retries: 30
  delay: 60
```

### Limit Parallelism
```yaml
- hosts: all
  serial: 5  # 5 hosts at a time
  
- hosts: all
  serial: "25%"  # 25% of hosts at a time
```

## Testing

### Syntax Check
```bash
ansible-playbook site.yml --syntax-check
```

### Check Mode (Dry Run)
```bash
ansible-playbook site.yml --check --diff
```

### Lint
```bash
ansible-lint site.yml
```

## Security

1. **Never commit secrets** - Use ansible-vault
2. **Limit sudo** - Use `become` only when needed
3. **Secure connections** - Use SSH keys, not passwords
4. **Audit changes** - Use `--diff` to see what changed
5. **Minimize permissions** - Principle of least privilege
