#!/usr/bin/env python3
"""
Track hotel prices and alert on drops via Amadeus API.
"""

import argparse
import json
import sys
import time
from datetime import datetime
from pathlib import Path
from typing import Optional

import requests

from auth import get_auth_header, get_base_url


# State file for tracked hotels
STATE_DIR = Path(__file__).parent.parent / "state"
TRACKED_FILE = STATE_DIR / "tracked.json"


def load_tracked() -> dict:
    """Load tracked hotels from state file."""
    if not TRACKED_FILE.exists():
        return {"hotels": []}
    
    try:
        with open(TRACKED_FILE) as f:
            return json.load(f)
    except json.JSONDecodeError:
        return {"hotels": []}


def save_tracked(data: dict) -> None:
    """Save tracked hotels to state file."""
    STATE_DIR.mkdir(parents=True, exist_ok=True)
    with open(TRACKED_FILE, "w") as f:
        json.dump(data, f, indent=2)


def make_request(url: str, params: dict, retries: int = 3) -> dict:
    """Make API request with retry logic for rate limits."""
    headers = get_auth_header()
    
    for attempt in range(retries):
        response = requests.get(url, headers=headers, params=params, timeout=30)
        
        if response.status_code == 429:
            wait = 2 ** attempt
            print(f"Rate limited, waiting {wait}s...", file=sys.stderr)
            time.sleep(wait)
            continue
        
        if response.status_code == 401:
            raise EnvironmentError(
                "Authentication failed. Check AMADEUS_API_KEY and AMADEUS_API_SECRET."
            )
        
        if response.status_code == 400:
            error = response.json().get("errors", [{}])[0]
            raise ValueError(f"Bad request: {error.get('detail', response.text)}")
        
        response.raise_for_status()
        return response.json()
    
    raise Exception("Max retries exceeded due to rate limiting")


def get_current_price(
    hotel_id: str,
    check_in: str,
    check_out: str,
    adults: int,
) -> Optional[float]:
    """Get current lowest price for a hotel."""
    base_url = get_base_url()
    
    params = {
        "hotelIds": hotel_id,
        "adults": adults,
        "checkInDate": check_in,
        "checkOutDate": check_out,
        "currency": "USD",
    }
    
    try:
        data = make_request(f"{base_url}/v3/shopping/hotel-offers", params)
        hotels = data.get("data", [])
        
        if not hotels:
            return None
        
        # Find lowest price across all offers
        lowest = None
        for hotel in hotels:
            for offer in hotel.get("offers", []):
                price = offer.get("price", {})
                total = float(price.get("total", 0))
                
                # Calculate per night
                ci = datetime.strptime(check_in, "%Y-%m-%d")
                co = datetime.strptime(check_out, "%Y-%m-%d")
                nights = (co - ci).days
                per_night = total / nights if nights > 0 else total
                
                if lowest is None or per_night < lowest:
                    lowest = per_night
        
        return lowest
    
    except Exception as e:
        print(f"Error fetching price for {hotel_id}: {e}", file=sys.stderr)
        return None


def add_hotel(
    hotel_id: str,
    check_in: str,
    check_out: str,
    adults: int,
    target_price: float,
) -> None:
    """Add hotel to tracking."""
    data = load_tracked()
    
    # Check if already tracked
    for hotel in data["hotels"]:
        if (hotel["hotelId"] == hotel_id and 
            hotel["checkin"] == check_in and 
            hotel["checkout"] == check_out):
            print(f"Hotel {hotel_id} already tracked for these dates.")
            return
    
    # Get current price
    current_price = get_current_price(hotel_id, check_in, check_out, adults)
    
    entry = {
        "hotelId": hotel_id,
        "checkin": check_in,
        "checkout": check_out,
        "adults": adults,
        "targetPrice": target_price,
        "lastPrice": current_price,
        "lastCheck": datetime.utcnow().isoformat() + "Z",
    }
    
    data["hotels"].append(entry)
    save_tracked(data)
    
    price_str = f"${current_price:.0f}" if current_price else "unknown"
    print(f"✅ Now tracking {hotel_id}")
    print(f"   Dates: {check_in} → {check_out}")
    print(f"   Current price: {price_str}/night")
    print(f"   Target: ${target_price:.0f}/night")


def remove_hotel(hotel_id: str) -> None:
    """Remove hotel from tracking."""
    data = load_tracked()
    
    original_count = len(data["hotels"])
    data["hotels"] = [h for h in data["hotels"] if h["hotelId"] != hotel_id]
    
    if len(data["hotels"]) < original_count:
        save_tracked(data)
        print(f"✅ Removed {hotel_id} from tracking")
    else:
        print(f"Hotel {hotel_id} not found in tracking list")


def list_tracked() -> None:
    """List all tracked hotels."""
    data = load_tracked()
    
    if not data["hotels"]:
        print("No hotels being tracked.")
        return
    
    print(f"📋 Tracking {len(data['hotels'])} hotel(s):\n")
    
    for hotel in data["hotels"]:
        hotel_id = hotel["hotelId"]
        checkin = hotel["checkin"]
        checkout = hotel["checkout"]
        target = hotel["targetPrice"]
        last_price = hotel.get("lastPrice")
        last_check = hotel.get("lastCheck", "never")
        
        price_str = f"${last_price:.0f}" if last_price else "unknown"
        
        print(f"🏨 {hotel_id}")
        print(f"   📅 {checkin} → {checkout}")
        print(f"   💰 Last: {price_str}/night | Target: ${target:.0f}/night")
        print(f"   🕐 Last checked: {last_check[:10] if last_check != 'never' else 'never'}")
        print()


def check_prices() -> None:
    """Check prices for all tracked hotels and report changes."""
    data = load_tracked()
    
    if not data["hotels"]:
        print("No hotels being tracked.")
        return
    
    alerts = []
    changes = []
    
    for hotel in data["hotels"]:
        hotel_id = hotel["hotelId"]
        checkin = hotel["checkin"]
        checkout = hotel["checkout"]
        adults = hotel.get("adults", 2)
        target = hotel["targetPrice"]
        last_price = hotel.get("lastPrice")
        
        # Skip if dates have passed
        if datetime.strptime(checkin, "%Y-%m-%d") < datetime.now():
            continue
        
        current = get_current_price(hotel_id, checkin, checkout, adults)
        
        if current is None:
            continue
        
        # Update state
        hotel["lastPrice"] = current
        hotel["lastCheck"] = datetime.utcnow().isoformat() + "Z"
        
        # Check for alerts
        if current <= target:
            alerts.append({
                "hotelId": hotel_id,
                "price": current,
                "target": target,
                "checkin": checkin,
                "checkout": checkout,
            })
        elif last_price and current < last_price:
            changes.append({
                "hotelId": hotel_id,
                "price": current,
                "was": last_price,
                "target": target,
            })
    
    save_tracked(data)
    
    # Output alerts (for notification systems)
    for alert in alerts:
        print(f"🚨 PRICE DROP: {alert['hotelId']} now ${alert['price']:.0f}/night "
              f"(target: ${alert['target']:.0f}) for {alert['checkin']} → {alert['checkout']}")
    
    for change in changes:
        print(f"📉 Price dropped: {change['hotelId']} now ${change['price']:.0f}/night "
              f"(was ${change['was']:.0f}, target: ${change['target']:.0f})")
    
    if not alerts and not changes:
        # Minimal output when nothing interesting
        print(f"✓ Checked {len(data['hotels'])} hotel(s) - no price changes")


def main():
    parser = argparse.ArgumentParser(description="Track hotel prices via Amadeus API")
    
    # Actions
    parser.add_argument("--add", action="store_true", help="Add hotel to tracking")
    parser.add_argument("--remove", action="store_true", help="Remove hotel from tracking")
    parser.add_argument("--list", action="store_true", help="List tracked hotels")
    parser.add_argument("--check", action="store_true", help="Check prices (for cron)")
    
    # Hotel details (for --add)
    parser.add_argument("--hotel", help="Hotel ID")
    parser.add_argument("--checkin", help="Check-in date (YYYY-MM-DD)")
    parser.add_argument("--checkout", help="Check-out date (YYYY-MM-DD)")
    parser.add_argument("--adults", type=int, default=2, help="Number of adults")
    parser.add_argument("--target", type=float, help="Target price per night")
    
    args = parser.parse_args()
    
    try:
        if args.add:
            if not all([args.hotel, args.checkin, args.checkout, args.target]):
                parser.error("--add requires --hotel, --checkin, --checkout, and --target")
            add_hotel(args.hotel, args.checkin, args.checkout, args.adults, args.target)
        
        elif args.remove:
            if not args.hotel:
                parser.error("--remove requires --hotel")
            remove_hotel(args.hotel)
        
        elif args.list:
            list_tracked()
        
        elif args.check:
            check_prices()
        
        else:
            parser.print_help()
    
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":
    main()
